<?php

namespace App\Http\Controllers\BackOffice\User;

use App\Http\Controllers\ApiController;
use App\Http\Controllers\BackOffice\Utilities\UtilitiesController;
use App\Models\user;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use JWTAuth;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\BackOffice\User\UserServices;


class UserController extends ApiController
{
    public function login(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'user_name' => 'required',
            'password' => 'required',
        ]);
        if ($validator->fails()) {
            return response()->json($validator->errors(), 422);
        }
        if (!$token = auth()->attempt($validator->validated())) {
            return ApiController::errorResponse('Wrong username or password !', 401);
        }
        $userInfo = user::leftjoin("clients", "clients.user_id", "=", "users.id")
            ->join('roles', 'roles.role_id', '=', 'users.user_role_id')
            ->where([
                ['clients.user_id', '=', null],
                ['users.user_name', 'like', $request->user_name],
            ])
            ->first();

        if ($userInfo) {
            $concatNameWithTime = time() . "" . $request->user_name;
            $hash = hash('sha256', $concatNameWithTime);
            User::where([
                ['id', '=', $userInfo->id],
            ])
                ->update([
                    'user_pushToken' => $request->notificationToken,
                    'user_identifier' => $hash
                ]);
            return ApiController::successResponse(([
                'access_token' => $token,
                'token_type' => 'bearer',
                'expires_in' => auth()->factory()->getTTL() * 180,
                'data' => $userInfo
            ]), 200);
        }
        return ApiController::errorResponse('Wrong username or password !', 401);
    }

    public function getUserInfo(Request $request)
    {
        $userInfo = auth()->user();
        $userInfo = user::join('useraccess', 'useraccess.user_id', '=', 'users.id')
            ->join('roles', 'roles.role_id', '=', 'users.user_role_id')
            ->leftjoin('companies', 'companies.company_id', '=', 'users.company_id')
            ->where([
                ['id', '=', $userInfo->id]
            ])
            ->first();


        $userInfo->company_logo = '/companies/images/' . $userInfo->company_id . '/' . $userInfo->company_logo;

        // return $request->header('Authorization');
        $rolesInfo = $this->getUserInfoFunction($request->header('Authorization'))->getData();
        if ($userInfo) {
            return ApiController::successResponse(([
                'access_token' => $request->header('Authorization'),
                'token_type' => 'bearer',
                'expires_in' => auth()->factory()->getTTL() * 180,
                'user_data' => $userInfo,
                "user_permissions" => $rolesInfo->data

            ]), 200);
        }
        return ApiController::errorResponse('Invalid user !', 401);
    }


    public function getUserInfoFunction($authToken)
    {
        if (!$authToken) {
            return ApiController::errorResponse('UnAuthorized !', 401);
        }
        $userId = auth()->user()->id;
        $permissions = array();
        if ($userId) {

            $userPermissions = DB::table('useraccess')
                ->where([
                    ['useraccess.user_id', '=', $userId]
                ])
                ->get();
            //return $userPermissions;
            foreach ($userPermissions as $access) {
                $objUserPermission = array();
                $objUserPermission['insertMode'] = $access->useraccess_insert;
                $objUserPermission['updateMode'] = $access->useraccess_update;
                $objUserPermission['deleteMode'] = $access->useraccess_delete;
                $objUserPermission['viewMode'] = $access->useraccess_view;
                $permissions[$access->useraccess_formcode] = $objUserPermission;
            }

            return ApiController::successResponse($permissions, 200);
        }
        return ApiController::errorResponse('UnAuthorized !', 401);
    }

    public function getUsers(Request $request)
    {
        $users = DB::table('users')
            ->join('roles', 'roles.role_id', '=', 'user_role_id')
            ->where('user_isActive', 1)
            ->get();
        if ($users)
            return ApiController::successResponse($users, 200);
        return ApiController::errorResponse('error !', 401);
    }

    public function register(Request $request)
    {
        $company_id = JWTAuth::user()->company_id;

        $validator = Validator::make($request->all(), [
            'userName' => 'required|string|min:2|max:100',

        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 400);
        }

        $checkUserExistance = User::where(
            [
                ['user_name', '=', $request->userName]
            ]
        )
            ->orwhere(
                [
                    ['user_email', '=', $request->userEmail]
                ]
            )
            ->first();

        if ($checkUserExistance) {
            return ApiController::errorResponse('usrname already taken!', 409);
        }
        $user = User::create([
            'user_name' => $request->userName,
            'user_isActive' => 1,
            'user_email' => $request->userEmail,
            'password' => Hash::make('0000'),
            'user_phone' => $request->userPhone,
            'user_role_id' => $request->userRoleId,
            'company_id' => $company_id,
            'name' => $request->name,
        ]);

        if ($user) {
            $getRoleSelectedPermissions = DB::table('roles_access')
                ->where([
                    ['role_id', '=', $request->userRoleId]
                ])
                ->get();

            for ($i = 0; $i < count($getRoleSelectedPermissions); $i++) {
                $row = $getRoleSelectedPermissions[$i];
                DB::table('useraccess')
                    ->insert([
                        'user_id' => $user->id,
                        'useraccess_formcode' => $row->roleaccess_formcode,
                        'useraccess_insert' => $row->roleaccess_insert,
                        'useraccess_update' => $row->roleaccess_update,
                        'useraccess_delete' => $row->roleaccess_delete,
                        'useraccess_view' => $row->roleaccess_view,
                    ]);
            }
        }
        return ApiController::successResponse('User successfully registered', 200);
    }

    public function updateUser(Request $request)
    {
        // $validator = Validator::make($request->all(), [
        //     'userName' => 'required|string|min:2|max:100',
        // ]);

        // if ($validator->fails()) {
        //     return response()->json($validator->errors(), 400);
        // }

        $checkUserExistance = User::where(
            [
                ['user_name', '=', $request->userName],
                ['id', '!=', $request->id]
            ]
        )
            ->orWhere(function ($query) use ($request) {
                $query->Where([
                    ['user_email', '=', $request->userEmail],
                    ['id', '!=', $request->id]
                ]);
            })
            ->first();

        if ($checkUserExistance) {
            return ApiController::errorResponse('usrname or email already taken!', 409);
        }
        $user = User::where([
            ['id', '=', $request->id],
        ])
            ->update([
                'user_name' => $request->userName,
                'user_email' => $request->userEmail,
                'user_phone' => $request->userPhone,
                'user_role_id' => $request->userRoleId,
                'name' => $request->name,
            ]);
        //  return $this->respondWithToken(auth()->attempt($validator->validated()));
        return ApiController::successResponse('User successfully registered', 200);
    }

    public function deactivateUser(Request $request)
    {
        // $validator = Validator::make($request->all(), [
        //     'id' => 'required',
        // ]);

        // if ($validator->fails()) {
        //     return response()->json($validator->errors(), 400);
        // }

        $checkUserExistance = User::where(
            [
                ['id', '=', $request->id]
            ]
        )
            ->first();

        if (!$checkUserExistance) {
            return ApiController::errorResponse('account not found', 409);
        }
        $user = User::where([
            ['id', '=', $request->id],
        ])
            ->update([
                'user_isActive' => 0,

            ]);
        //  return $this->respondWithToken(auth()->attempt($validator->validated()));
        return ApiController::successResponse('User successfully Deactivate', 200);
    }

    public function fetchCustomUserRoles(Request $request)
    {
        $userPermissions = DB::table('useraccess')
            ->where([
                ['useraccess.user_id', '=', $request->userId]
            ])
            ->get();
        if (count($userPermissions)) {
            return ApiController::successResponse($userPermissions, 200);
        }
        $getUserRole = user::where('id', '=', $request->userId)->select('user_role_id')->first();

        if ($getUserRole) {
            $roles = DB::table('roles_access')->where('role_id', '=', $getUserRole->user_role_id)->get();

            for ($i = 0; $i < count($roles); $i++) {
                $row = $roles[$i];
                DB::table('useraccess')
                    ->insert([
                        'user_id' => $request->userId,
                        'useraccess_formcode' => $row->roleaccess_formcode,
                        'useraccess_insert' => $row->roleaccess_insert,
                        'useraccess_update' => $row->roleaccess_update,
                        'useraccess_delete' => $row->roleaccess_delete,
                        'useraccess_view' => $row->roleaccess_view,
                    ]);
            }
        }
        $userPermissions = DB::table('useraccess')
            ->where([
                ['useraccess.user_id', '=', $request->userId]
            ])
            ->get();

        return ApiController::successResponse($userPermissions, 200);
    }

    public function insertCustomRoleByUser(Request $request)
    {
        $requestContent = array();
        $requestContent = json_decode($request->getContent(), true);
        $screens = $requestContent['screens']; // global screens
        $userPermissions = $requestContent['userPermissions']; // insert update delete view
        $user = $requestContent['user']; // user id

        if ($user && $screens && $userPermissions) {
            for ($i = 0; $i < count($screens); $i++) {
                $row = $screens[$i]; // screenName // screenCode
                $permission = $userPermissions[$row['screenCode']];
                $checkAccessExistance = DB::table('useraccess')
                    ->where([
                        ['useraccess.user_id', '=', $user],
                        ['useraccess.useraccess_formcode', '=', $row['screenCode']]
                    ])
                    ->first(); // check if role is already created
                if ($checkAccessExistance) // if true update
                {
                    DB::table('useraccess')
                        ->where([
                            ['useraccess.user_id', '=', $user],
                            ['useraccess.useraccess_formcode', '=', $row['screenCode']]
                        ])
                        ->update([
                            'useraccess_insert' => $permission['insertMode'],
                            'useraccess_update' => $permission['updateMode'],
                            'useraccess_delete' => $permission['deleteMode'],
                            'useraccess_view' => $permission['viewMode'],
                        ]);
                } else { // else insert new role
                    DB::table('useraccess')
                        ->insert([
                            'user_id' => (int)$user,
                            'useraccess_formcode' => $row['screenCode'],
                            'useraccess_insert' => $permission['insertMode'],
                            'useraccess_update' => $permission['updateMode'],
                            'useraccess_delete' => $permission['deleteMode'],
                            'useraccess_view' => $permission['viewMode'],
                        ]);
                }
            }
            return ApiController::successResponse($userPermissions, 200);
        }
        return ApiController::errorResponse("Invalid request", 422);
    }

    public function forgotPassword(Request $request)
    {

        $utilitiesController = new UtilitiesController();
        $validator = Validator::make($request->all(), [
            'email' => 'required|email',
        ]);
        if ($validator->fails()) {
            return ApiController::errorResponse($validator->errors(), 422);
        }


        $checkCurrentEmail = DB::table('users')
            ->where([
                ['users.user_email', '=', $request->email]
            ])
            ->select('users.user_email', 'users.id', 'user_identifier')
            ->first();

        if ($checkCurrentEmail) {

            // if ($checkCurrentEmail->user_status != 'approved') {
            //     return ApiController::errorResponse("Permission blocked!", 422);
            // }
            $link = 'Please click on the link below :' . config("app.SITE_URL") . '/reset-password/' . $checkCurrentEmail->user_identifier;
            $email = $utilitiesController->sendMail($link, 'Forgot Password', $checkCurrentEmail->user_email);

            return ApiController::successResponse($checkCurrentEmail, 200);
        }
        return ApiController::errorResponse("Email does not exist in our database!", 422);
    }

    public function resetUserPassword(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'password' => 'required',
        ]);

        if ($validator->fails()) {
            return ApiController::errorResponse($validator->errors(), 422);
        }
        $concatNameWithTime = time() . "" . $request->user_name;
        $hash = hash('sha256', $concatNameWithTime);

        $resetPassword = DB::table('users')
            ->where([
                ['user_identifier', '=', $request->identifier]
            ])
            ->update([
                'password' => bcrypt($request->password),
                'user_identifier' => $hash
            ]);

        if ($resetPassword) {
            return ApiController::successResponse($resetPassword, 200);
        }
        return ApiController::errorResponse("Unauthorized", 401);
    }

    public function changeUserPassword(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'currentPassword' => 'required',
            'newPassword' => 'required',
        ]);
        if ($validator->fails()) {
            return ApiController::errorResponse($validator->errors(), 422);
        }
        $payload = array();
        $payload['currentPassword'] = $request->currentPassword;
        $payload['newPassword'] = $request->newPassword;

        $userService = new UserServices();
        return $userService->changePasswordService($payload);
    }
}
