<?php

namespace App\Http\Controllers\BackOffice\GoodsIn;

use App\enums\NotificationTopics;
use App\Http\Controllers\ApiController;
use App\Http\Controllers\BackOffice\Common\CommonServices;
use App\Http\Controllers\BackOffice\History\HistoryController;
use App\Http\Controllers\BackOffice\Notifications\NotificationsController;
use App\Http\Controllers\BackOffice\Utilities\UtilitiesController;
use App\Models\clientItem;
use App\Models\clientSupplier;
use App\Models\companies;
use App\Models\country;
use App\Models\items;
use App\Models\location;
use App\Models\notifications;
use App\Models\sequence;
use App\Models\suppliers;
use App\Models\transactionDt;
use App\Models\transactionDtsn;
use App\Models\transactionDtSnTmp;
use App\Models\transactiondttmp;
use App\Models\transactions;
use App\Models\transactiontmp;
use App\Models\TransactiontmpPalets;
use Carbon\Carbon;
use Exception;
use File;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use JWTAuth;

class GoodsInController extends ApiController
{

    public function createTransactionTmpGoodsIn(Request $request)
    {
        $utilitiesController = new UtilitiesController();
        $request = $utilitiesController->checkGoodsInHeaderModules($request);
        $checkCompany = companies::first();
        $validator = Validator::make($request->all(), [
            'transactionDate' => 'required',
            'transactionType' => 'required',
            'transactionBillOfEntry' => $checkCompany->company_enableDraftWithoutDEC == 0 ? 'required' : '',
        ]);
        if ($validator->fails()) {
            return response()->json($validator->errors(), 422);
        }
        $user = auth()->user();
        $transactionType = strToUpper(str_replace(' ', '_', $request->transactionType));
        $companyId = companies::first()->company_id;
        $currentDateTime = Carbon::now();
        $formattedTime = $currentDateTime->format('H:i');

        $transactionTmp = array(
            'transactiontmp_date' => $utilitiesController->convertTodayToTimetsamp($request->transactionDate . "" . date('H:i:s')),
            'transaction_time' => $formattedTime,
            'transactiontmp_type' => $transactionType,
            'transactiontmp_remark' => $request->transactionRemark,
            'transactiontmp_billOfEntry' => $request->transactionBillOfEntry,
            'user_id' => $user->id,
            'forwarder_id' => $request->orderForwarder,
            'clientCustomer_id' => $request->transactionCustomer,
            'company_id' => $companyId,
            'transactiontmp_currencyId' => $request->transactionCurrency,
            'transactiontmp_rate' => $request->transactionRate,
            'transactiontmp_shipmentRemark' => $request->transactionShipmentRemark,
            'transactiontmp_supplierId' => $request->transactionSupplier,
            'transactiontmp_supplierInvoice' => $request->transactionSupplierInvoice,
            'transactiontmp_BOE' => $request->transactionBOE != '' ? $request->transactionBOE : '-',
            'transactiontmp_clientId' => isset($request->transactionClient) ? $request->transactionClient : '-'
        );

        return GoodsInController::createTransactionTmp($transactionTmp, $request->transactionTmpId,);
    }

    public function createTransactionTmp($transactionTmp, $transactionTmpId)

    {
        $goodsInFunctions = new GoodsInFunctions();
        $checkOriginalTransactionExistance = $goodsInFunctions->getOriginalTransactionExistance($transactionTmpId);
        $getCurrentTransaction = $goodsInFunctions->getCurrentTransaction($transactionTmpId);

        $goodsInFunctions->updateTransactionDts($getCurrentTransaction, $transactionTmp, $transactionTmpId);

        if ($checkOriginalTransactionExistance) {
            $transactionTmp['transactiontmp_reference'] = $checkOriginalTransactionExistance->transaction_reference;
            $transactionTmp['transaction_time'] = $checkOriginalTransactionExistance->transaction_time;
        }

        $result = $goodsInFunctions->checkBOEDECEXist($transactionTmpId, $transactionTmp);
        if ($result->getData()->code == 422) {
            return $result;
        }

        $updatedTransactionTmp = $goodsInFunctions->updateOrCreateTransactionTmp($transactionTmpId, $transactionTmp);
        $transactionTmpAfterUpdate = $goodsInFunctions->getTransactionTmpAfterUpdate($updatedTransactionTmp);

        $goodsInFunctions->updateTransactionDttmp($transactionTmpAfterUpdate);

        if ($transactionTmpAfterUpdate) {
            return ApiController::successResponse($transactionTmpAfterUpdate, 200);
        }

        return ApiController::errorResponse($transactionTmpAfterUpdate, 422);
    }

    public function removeTransactionDtTmp(Request $request)
    {

        $transactionDt = transactiondttmp::join('transactiontmps', 'transactiontmps.transactiontmp_id', 'transactiondttmps.transactiontmp_id')
            ->where('transactiondttmps.transactiondttmp_id', '=', $request->transactionDtTmpId)
            ->first();


        if (isset($transactionDt->transactiontmp_reference)) {
            $removeTransactionDtTmp = transactiondttmp::where('transactiondttmp_id', '=', $request->transactionDtTmpId)->update([
                'state' => 'deleted'
            ]);
        } else {
            $removeTransactionDtTmp = transactiondttmp::where('transactiondttmp_id', '=', $request->transactionDtTmpId)->delete();
        }


        if ($removeTransactionDtTmp) {
            $transactionDtSumQty = transactiondttmp::where('transactiondttmps.transactiontmp_id', '=', $transactionDt->transactiontmp_id)
                ->select('transactiondttmps.item_quantity')
                ->selectRaw(' SUM(transactiondttmps.item_quantity) as totalQty')
                ->selectRaw(' SUM(transactiondttmps.item_quantity * transactiondttmps.transactiondttmp_price) as totalPrice')
                ->first();


            $transactionTotalItem = transactiondttmp::where('transactiondttmps.transactiontmp_id', '=', $transactionDt->transactiontmp_id)
                ->max('transactiondttmps.raw_in_grid');


            transactiontmp::where('transactiontmps.transactiontmp_id', '=', $transactionDt->transactiontmp_id)
                ->update(
                    [
                        "transactiontmp_totalQty" => $transactionDtSumQty->totalQty,
                        "transaction_totalPrice" => $transactionDtSumQty->totalPrice,
                        "transaction_totalItem" => $transactionTotalItem + 1,
                    ]
                );

            if ($transactionDt->item_qtyPerPack) {

                $transactionDtSumNoCartoon = transactiondttmp::where([
                    ['transactiontmp_id', $transactionDt->transactiontmp_id],
                    ['raw_in_grid', $transactionDt->raw_in_grid]

                ])->sum('item_quantity');

                $newQty = ceil($transactionDtSumNoCartoon / $transactionDt->item_qtyPerPack);


                $updateTransactionDtSumNoCartoon = transactiondttmp::where([
                    ['transactiontmp_id', $transactionDt->transactiontmp_id],
                    ['raw_in_grid', $transactionDt->raw_in_grid]
                ])
                    ->update(["transactiondttmp_nbOfCarton" => $newQty]);
            }


            return ApiController::successResponse($removeTransactionDtTmp, 200);
        }
        return ApiController::errorResponse($removeTransactionDtTmp, 422);
    }

    public function removeOriginalTransaction(Request $request)
    {
        $user = auth()->user();
        $date = date('Y-m-d H:i:s');
        $utilitiesController = new UtilitiesController();

        $checkExsistance = transactiontmp::where('transactiontmp_id', '=', $request->transactionId)
            ->join('users', 'users.id', '=', 'transactiontmps.user_id')
            ->join('clients', 'clients.client_id', '=', 'transactiontmps.transactiontmp_clientId')
            ->leftjoin('client_customers', 'client_customers.customer_id', '=', 'transactiontmps.clientCustomer_id')
            ->leftjoin('client_forwarders', 'client_forwarders.clientforwarder_id', '=', 'transactiontmps.forwarder_id')
            ->orderBy('transactiontmp_date', 'asc')
            ->first();

        if ($checkExsistance) {
            $type = "Goods In";
            if ($checkExsistance->transactiontmp_type == "DO") {
                $type = "Delivery Order";
            }

            return ApiController::errorResponse($type . '  exists in drafts !(By : ' . $checkExsistance->user_name . ')', 422);
        }


        $checkValidity = transactionDt::join('transaction_dts as dt', 'dt.id_reference_dt', 'transaction_dts.transactiondt_id')
            ->where([
                ['transaction_dts.transaction_id', '=', $request->transactionId],
            ])->get();


        if (count($checkValidity) == 0) {
            $getDtData = transactionDt::where([
                ['transaction_dts.transaction_id', '=', $request->transactionId],
            ])->get();
            $historyController = new HistoryController();

            $transactions = transactions::where('transaction_id', '=', $request->transactionId);
            $removeTransactions = $transactions->first()->toarray();
            $removeTransactions['user_modified'] = $user->id;
            $removeTransactions['date_modified'] = $date;


            $path = public_path() . '/documents/' . $removeTransactions['transaction_type'] . '/' . $request->transactionId . '/' . $request->file;

            if (file_exists($path)) {
                $deleteFile = File::deleteDirectory($path);
            }
            $deletedd = $transactions->first();

            if (isset($deletedd->salesjv_id)) {
                $config = array(
                    "salesjvId" => $deletedd->salesjv_id,
                );
                $utilitiesController->connectDataToErp("deleteInvoiceFromLogisticom", $config);
            }
            $historyController->insertTransactionHistoryDataByRef(
                'deleted',
                $deletedd->transaction_reference,
                auth()->user()->id,
                $date,
                $deletedd->transaction_type
            );
            $transactions = $transactions->delete();
            $transactionDtArray = null;


            $transferTrans = transactions::where('relatedTransferId', $request->transactionId)
                ->first();
            if ($transferTrans) {

                transactionDt::where('transaction_id', $transferTrans->transaction_id)->delete();
                transactions::where('transaction_id', $transferTrans->transaction_id)->delete();
            }


            return ApiController::successResponse($transactions, 200);
        }

        return ApiController::successResponse('unable to delete', 204);
    }

    public function getNextReference(Request $request)
    {
        $sequence = sequence::where('sequence_transactionType', 'like', $request->transactionType)
            ->max('sequence_transactionRef') + 1;
        if (!$sequence) {
            return ApiController::errorResponse($sequence, 422);
        }
        return ApiController::successResponse($sequence, 200);
    }

    public function insertNewTransactionSave2(Request $request)

    {
        $checkTransactionTmpExistance = transactiontmp::where('transactiontmp_id', '=', $request->transactionTmpId)
            ->first();
        $getSequence = sequence::where('sequence_transactionType', $checkTransactionTmpExistance->transactiontmp_type)->first();
        if (!$getSequence) {
            return ApiController::errorResponse($getSequence, 422);
        }
        $sequenceNumber = 0;
        if (!isset($checkTransactionTmpExistance->transactiontmp_reference)) {
            $sequenceNumber = $getSequence->sequence_transactionRef + 1;
            sequence::where('sequence_transactionType', 'like', $checkTransactionTmpExistance->transactiontmp_type)->increment('sequence_transactionRef', 1);
        }

        DB::beginTransaction();
        $transaction = new transactions();
        $transaction->transaction_reference = isset($checkTransactionTmpExistance->transactiontmp_reference) ? $checkTransactionTmpExistance->transactiontmp_reference : $sequenceNumber;
        $transaction->transaction_id = $checkTransactionTmpExistance->transactiontmp_id;
        $transaction->transaction_date = $checkTransactionTmpExistance->transactiontmp_date;
        $transaction->transaction_clientId = $checkTransactionTmpExistance->transactiontmp_clientId;
        $transaction->transaction_type = $checkTransactionTmpExistance->transactiontmp_type;
        $transaction->transaction_remark = $checkTransactionTmpExistance->transactiontmp_remark;
        $transaction->transaction_mobileReference = $checkTransactionTmpExistance->transactiontmp_mobileReference;
        $transaction->transaction_shipmentReference = $checkTransactionTmpExistance->transactiontmp_shipmentReference;
        $transaction->transaction_billOfEntry = $checkTransactionTmpExistance->transactiontmp_billOfEntry;
        $transaction->user_id = $checkTransactionTmpExistance->user_id;
        $transaction->transaction_typeOfDeliveryId = $checkTransactionTmpExistance->transactiontmp_typeOfDeliveryId;
        $transaction->forwarder_id = $checkTransactionTmpExistance->forwarder_id;
        $transaction->clientCustomer_id = $checkTransactionTmpExistance->clientCustomer_id;
        $transaction->transaction_driverName = $checkTransactionTmpExistance->transactiontmp_driverName;
        $transaction->transaction_vehicleNumber = $checkTransactionTmpExistance->transactiontmp_vehicleNumber;
        $transaction->transaction_exitPoint = $checkTransactionTmpExistance->transactiontmp_exitPoint;
        $transaction->transaction_destinationPoint = $checkTransactionTmpExistance->transactiontmp_destinationPoint;
        $transaction->company_id = $checkTransactionTmpExistance->company_id;
        $transaction->transaction_paymentMethodId = $checkTransactionTmpExistance->transactiontmp_paymentMethodId;
        $transaction->transaction_customBillTypeId = $checkTransactionTmpExistance->transactiontmp_customBillTypeId;
        $transaction->transaction_currencyId = $checkTransactionTmpExistance->transactiontmp_currencyId;
        $transaction->transaction_rate = $checkTransactionTmpExistance->transactiontmp_rate;
        $transaction->transaction_shipmentTypeId = $checkTransactionTmpExistance->transactiontmp_shipmentTypeId;
        $transaction->transaction_shipById = $checkTransactionTmpExistance->transactiontmp_shipById;
        $transaction->transaction_shipmentRemark = $checkTransactionTmpExistance->transactiontmp_shipmentRemark;
        $transaction->transaction_supplierId = $checkTransactionTmpExistance->transactiontmp_supplierId;
        $transaction->transaction_supplierInvoice = $checkTransactionTmpExistance->transactiontmp_supplierInvoice;
        $transaction->transaction_totalQty = $checkTransactionTmpExistance->transactiontmp_totalQty;
        $transaction->transaction_totalPallet = $checkTransactionTmpExistance->transactiontmp_totalPallet;
        $transaction->transaction_totalPrice = $checkTransactionTmpExistance->transaction_totalPrice;
        $transaction->transaction_totalItem = $checkTransactionTmpExistance->transaction_totalItem;
        $transaction->transaction_BOE = $checkTransactionTmpExistance->transactiontmp_BOE;
        $transaction->transaction_released_DEC = $checkTransactionTmpExistance->transaction_released_DEC;
        $transaction->save();

        DB::commit();

        $transactionDtsTmp = transactiondttmp::join('items', 'items.item_id', '=', 'transactiondttmps.item_id')
            ->select('transactiondttmps.*', 'items.item_hasSerial')
            ->where('transactiontmp_id', '=', $request->transactionTmpId)
            ->get();
        foreach ($transactionDtsTmp as $transactionDtTmp) {
            if ($transactionDtTmp->state == 'deleted') continue;

            $transactiondt = new transactionDt();
            $transactiondt->transactiondt_id = $transactionDtTmp->transactiondttmp_id;
            $transactiondt->transaction_id = $transaction->transaction_id;
            $transactiondt->item_id = $transactionDtTmp->item_id;
            $transactiondt->client_id = $transactionDtTmp->client_id;
            $transactiondt->item_quantity = $transactionDtTmp->item_quantity;
            $transactiondt->supplier_id = $transactionDtTmp->supplier_id;
            $transactiondt->supplier_invoice_number = $transactionDtTmp->supplier_invoice_number;
            $transactiondt->transactiondt_locationid = $transactionDtTmp->transactiondttmp_locationid;
            $transactiondt->transactiondt_countryOfOrigin = $transactionDtTmp->transactiondttmp_countryOfOrigin;
            $transactiondt->transactiondt_price = $transactionDtTmp->transactiondttmp_price;
            $transactiondt->transactiondt_priceBase = $transactionDtTmp->transactiondttmp_priceBase;
            $transactiondt->id_reference_dt = $transactionDtTmp->id_reference_dt;
            $transactiondt->transactiondt_billOfEntry = $transactionDtTmp->transactiontmpdt_billOfEntry;
            $transactiondt->raw_in_grid = $transactionDtTmp->raw_in_grid;
            $transactiondt->item_weight = $transactionDtTmp->item_weight;
            $transactiondt->item_unit = $transactionDtTmp->item_unit;
            $transactiondt->item_qtyPerPack = $transactionDtTmp->item_qtyPerPack;
            $transactiondt->item_pack = $transactionDtTmp->item_pack;
            $transactiondt->group_id = $transactionDtTmp->group_id;
            $transactiondt->palet_no = $transactionDtTmp->palet_no;
            $transactiondt->transactiondt_nbOfCarton = $transactionDtTmp->transactiondttmp_nbOfCarton;
            $transactiondt->transactiondt_cellMerge = $transactionDtTmp->transactiondttmp_cellMerge;
            $transactiondt->transactiondt_BOE = $transactionDtTmp->transactiontmpdt_BOE;
            $transactiondt->transactiondt_fprice = $transactionDtTmp->transactiondttmp_fprice;
            $transactiondt->transactiondt_fpriceBase = $transactionDtTmp->transactiondttmp_fpriceBase;
            $transactiondt->hscode = $transactionDtTmp->hscode;
            $transactiondt->transactiondt_item_description = $transactionDtTmp->transactiondt_item_description;
            $transactiondt->total_price = $transactionDtTmp->total_price;
            $transactiondt->total_weight = $transactionDtTmp->total_weight;
            $transactiondt->save();


            if ($transactionDtTmp->item_id) {
                $this->createClientItems($transactionDtTmp, $transactionDtTmp->client_id);
            }
        }
        transactiontmp::where('transactiontmp_id', '=', $request->transactionTmpId)->delete();
    }

    public function createClientItems($transactionDtTmp, $clientId)
    {

        $getItemDetails = items::where('item_id', '=', $transactionDtTmp->item_id)->first();

        $checkIfItemExist = clientItem::where([
            ['item_id', '=', $getItemDetails->item_id],
            ['item_related_client_id', '=', $clientId]
        ])
            ->first();

        if (!$checkIfItemExist && $getItemDetails) {
            $itemArray = array(
                'item_id' => $getItemDetails->item_id,
                'item_code' => $getItemDetails->item_code,
                'item_description' => $getItemDetails->item_description,
                'item_barcode' => $getItemDetails->item_barcode,
                'item_related_client_id' => $clientId,
                'item_hasSerial' => $getItemDetails->item_hasSerial,
                'item_hscode' => $getItemDetails->item_hscode,
                'item_weight' => $getItemDetails->item_weight,
                'item_unit' => $getItemDetails->item_unit,
                'item_qtyPerPack' => $getItemDetails->item_qtyPerPack,
                'item_pack' => $getItemDetails->item_pack,
                'group_id' => $getItemDetails->group_id,
            );
            clientItem::create($itemArray);
        }
    }

    public function insertNewTransaction(Request $request)
    {
        $date = date('Y-m-d H:i:s');
        $historyController = new HistoryController();
        $utilitiesController = new UtilitiesController();
        $hitoryArray = array();
        $state = "Insert";
        // retrieve the current transaction tmp
        $checkTransactionTmpExistance = transactiontmp::where('transactiontmp_id', '=', $request->transactionTmpId)
            ->leftjoin('clients', 'clients.client_id', 'transactiontmps.transactiontmp_clientId')
            ->select('transactiontmps.*', 'clients.client_firstName')
            ->first();
        // retrieve the current transaction dt tmp
        $transactionDtsTmp = transactiondttmp::join('items', 'items.item_id', '=', 'transactiondttmps.item_id')
            ->select('transactiondttmps.*', 'items.item_hasSerial')
            ->where([
                ['transactiontmp_id', '=', $request->transactionTmpId],
                ['state', '!=', 'deleted']
            ])
            ->get();

        $invalidTransactionsIndex = array();
        $navigationTypeOUT = $checkTransactionTmpExistance->transactiontmp_type == 'GOODS_IN' ? 'DO' : 'RMA_OUT';

        for ($i = 0; $i < count($transactionDtsTmp); $i++) {
            $quantity = $transactionDtsTmp[$i]->item_quantity;

            $checkNbrOfSerialsCreatedTmp = transactionDtSnTmp::where([
                ['transactiondttmp_id', '=', $transactionDtsTmp[$i]->transactiondttmp_id],
            ])
                ->where(function ($query) {
                    $query->where('status', '!=', 'Delete')
                        ->orWhereNull('status');
                })
                ->get();


            $checkNewQtyEntry = DB::table('transactiondttmps')
                ->join('items', 'items.item_id', 'transactiondttmps.item_id')
                ->join('transactiontmps', 'transactiontmps.transactiontmp_id', 'transactiondttmps.transactiontmp_id')
                ->select('transactiondttmps.transactiondttmp_id', 'transactiondttmps.transactiontmpdt_billOfEntry', 'transactiondttmps.item_id', 'client_id', 'item_quantity', 'items.*',)
                ->selectRaw(' SUM(item_quantity)  as item_quantity')
                ->where([
                    ['transactiontmp_type', $checkTransactionTmpExistance->transactiontmp_type],
                    ['transactiondttmps.item_id', $transactionDtsTmp[$i]->item_id],
                    ['transactiondttmps.transactiontmpdt_billOfEntry', $transactionDtsTmp[$i]->transactiontmpdt_billOfEntry],
                    ['transactiondttmps.transactiondttmp_countryOfOrigin', $transactionDtsTmp[$i]->transactiondttmp_countryOfOrigin]
                ])
                ->groupBy('transactiondttmps.item_id')
                ->first();
            $QtyEnter = 0;
            $QtyEnter = isset($checkNewQtyEntry->item_quantity) ? $checkNewQtyEntry->item_quantity : 0;


            $checkQtyExit = DB::table('transaction_dts')
                ->join('items', 'items.item_id', 'transaction_dts.item_id')
                ->join('transactions', 'transactions.transaction_id', 'transaction_dts.transaction_id')
                ->select('transaction_dts.transactiondt_billOfEntry', 'transaction_dts.item_id', 'client_id', 'item_quantity', 'items.*')
                ->selectRaw(' SUM(item_quantity)  as item_quantity')
                ->where([
                    ['transaction_dts.id_reference_dt', $checkNewQtyEntry->transactiondttmp_id],
                ])
                ->groupBy('transaction_dts.item_id')
                ->first();

            $QtyExit = isset($checkQtyExit->item_quantity) ? $checkQtyExit->item_quantity : 0;

            if (($transactionDtsTmp[$i]->item_hasSerial && $quantity != count($checkNbrOfSerialsCreatedTmp))) {
                array_push($invalidTransactionsIndex, $transactionDtsTmp[$i]);
            } else if ((int)$QtyExit > $QtyEnter) {
                array_push($invalidTransactionsIndex, $transactionDtsTmp[$i]);
            }
        }

        if (count($invalidTransactionsIndex)) {
            return ApiController::errorResponse($invalidTransactionsIndex, 422);
        }

        // delete previous transactions dts
        $previousDtData = null;
        if ($request->transactionId) {
            $previousDtData = transactiondt::where('transaction_id', '=', $request->transactionId)->get();
            transactiondt::where('transaction_id', '=', $request->transactionId)->delete();
        }
        $newTransaction = null;
        $getSequence = null;

        if ($checkTransactionTmpExistance) {
            //return ApiController::errorResponse($transactionDtsTmp,422);
            $getSequence = sequence::where('sequence_transactionType', 'like', $checkTransactionTmpExistance->transactiontmp_type)->first();
            if (!$getSequence) {
                return ApiController::errorResponse($getSequence, 422);
            }

            $sequenceNumber = $getSequence->sequence_transactionRef + 1;
            $goodsInSequence = $checkTransactionTmpExistance->transactiontmp_reference ? $checkTransactionTmpExistance->transactiontmp_reference : $sequenceNumber;


            $transaction = array(
                'transaction_reference' => $goodsInSequence,
                'transaction_id' => $checkTransactionTmpExistance->transactiontmp_id,
                'transaction_time' => $checkTransactionTmpExistance->transaction_time,
                'transaction_date' => $checkTransactionTmpExistance->transactiontmp_date,
                'transaction_clientId' => $checkTransactionTmpExistance->transactiontmp_clientId,
                'transaction_type' => $checkTransactionTmpExistance->transactiontmp_type,
                'transaction_remark' => $checkTransactionTmpExistance->transactiontmp_remark,
                'transaction_mobileReference' => $checkTransactionTmpExistance->transactiontmp_mobileReference,
                'transaction_shipmentReference' => $checkTransactionTmpExistance->transactiontmp_shipmentReference,
                'transaction_billOfEntry' => $checkTransactionTmpExistance->transactiontmp_billOfEntry,
                'user_id' => $checkTransactionTmpExistance->user_id,
                'transaction_typeOfDeliveryId' => $checkTransactionTmpExistance->transactiontmp_typeOfDeliveryId,
                'forwarder_id' => $checkTransactionTmpExistance->forwarder_id,
                'clientCustomer_id' => $checkTransactionTmpExistance->clientCustomer_id,
                'transaction_driverName' => $checkTransactionTmpExistance->transactiontmp_driverName,
                'transaction_vehicleNumber' => $checkTransactionTmpExistance->transactiontmp_vehicleNumber,
                'transaction_exitPoint' => $checkTransactionTmpExistance->transactiontmp_exitPoint,
                'transaction_destinationPoint' => $checkTransactionTmpExistance->transactiontmp_destinationPoint,
                'company_id' => $checkTransactionTmpExistance->company_id,
                'transaction_paymentMethodId' => $checkTransactionTmpExistance->transactiontmp_paymentMethodId,
                'transaction_customBillTypeId' => $checkTransactionTmpExistance->transactiontmp_customBillTypeId,
                'transaction_currencyId' => $checkTransactionTmpExistance->transactiontmp_currencyId,
                'transaction_rate' => $checkTransactionTmpExistance->transactiontmp_rate,
                'transaction_shipmentTypeId' => $checkTransactionTmpExistance->transactiontmp_shipmentTypeId,
                'transaction_shipById' => $checkTransactionTmpExistance->transactiontmp_shipById,
                'transaction_shipmentRemark' => $checkTransactionTmpExistance->transactiontmp_shipmentRemark,
                'transaction_supplierId' => $checkTransactionTmpExistance->transactiontmp_supplierId,
                'transaction_supplierInvoice' => $checkTransactionTmpExistance->transactiontmp_supplierInvoice,
                'transaction_totalQty' => $checkTransactionTmpExistance->transactiontmp_totalQty,
                'transaction_totalPallet' => $checkTransactionTmpExistance->transactiontmp_totalPallet,
                'transaction_totalPrice' => $checkTransactionTmpExistance->transaction_totalPrice,
                'transaction_totalItem' => $checkTransactionTmpExistance->transaction_totalItem,
                'transaction_BOE' => $checkTransactionTmpExistance->transactiontmp_BOE,
                'transaction_released_DEC' => $checkTransactionTmpExistance->transaction_released_DEC,


            );
            $clients = array(
                'table' => 'clients',
                'column' => 'client_id',
                'value' => $checkTransactionTmpExistance->transactiontmp_clientId,
                'selectedValue' => 'client_firstname',
                'replaceColumn' => 'transaction_clientId'
            );
            $customers = array(
                'table' => 'client_customers',
                'column' => 'customer_id',
                'value' => $checkTransactionTmpExistance->clientCustomer_id,
                'selectedValue' => 'customer_name',
                'replaceColumn' => 'clientCustomer_id'
            );

            array_push($hitoryArray, $clients);
            array_push($hitoryArray, $customers);

            if ($checkTransactionTmpExistance && !$checkTransactionTmpExistance->transactiontmp_reference) {

                $newTransaction = transactions::create($transaction);
                $notificationController = new NotificationsController();
                $notificationController->sendNotificationsByTopic(auth()->user()->id, NotificationTopics::GI_RECEIVED, $newTransaction->transaction_id, $newTransaction->transaction_reference);

                $utilitiesController->createErpProject($newTransaction->transaction_billOfEntry, $newTransaction->transaction_type, $checkTransactionTmpExistance->client_firstName, $checkTransactionTmpExistance->transactiontmp_id);
                $transaction['transaction_id'] = $newTransaction->transaction_id;
                $transaction['user_modified'] = auth()->user()->id;
                $transaction['modification_type'] = 'Insert';
                $transaction['date_modified'] = $date;

                $notificationToken = DB::table('users')
                    ->join('clients', 'clients.user_id', 'id')
                    ->where([
                        ["clients.client_id", $checkTransactionTmpExistance->transactiontmp_clientId],
                    ])
                    ->first();


                $notificationKey = "";
                $type = "GoodsIn";
                $notificationKey = "goodsIn";
                $ref = $goodsInSequence;
                $message = $type . ' Ref: ' . $ref . ' was created';
                if ($checkTransactionTmpExistance->transactiontmp_type == 'RMA_IN') {
                    $type = "RMAIN";
                    $notificationKey = "rmaIn";
                }


                ///notification to client
                if (isset($notificationToken->user_pushToken)) {
                    $notificationArray = array(
                        'title' => $type,
                        'message' => $message,
                        'seen' => 0,
                        'related_userid' => $checkTransactionTmpExistance->user_id,
                        'trans_id' => $newTransaction->transaction_id,
                        'key' => $newTransaction->transaction_id,
                        'code' => "GoodsIn_Created",
                        'notificationToken' => $notificationToken->user_pushToken,
                        'transType' => $notificationKey
                    );
                    $utilitiesController->sendNotification($notificationArray);
                }
                $admins = DB::table('users')
                    ->join('useraccess', 'useraccess.user_id', 'users.id')
                    ->where([
                        ["useraccess.useraccess_formcode", $notificationKey],
                    ])
                    ->get();

                foreach ($admins as $admindt) {

                    if ($admindt->user_pushToken) {
                        $notificationArray = array(
                            'title' => $type,
                            'message' => 'New ' . $type . ' Transaction was created',
                            'seen' => 0,
                            'related_userid' => $admindt->id,
                            'trans_id' => $newTransaction->transaction_id,
                            'key' => $newTransaction->transaction_id,
                            'code' => "GoodsIn_Created",
                            'notificationToken' => $admindt->user_pushToken,
                            'transType' => $notificationKey
                        );
                        $utilitiesController->sendNotification($notificationArray);
                    }
                }
            } else {
                transactions::where('transaction_id', '=', $request->transactionId)->update($transaction);
                $notificationController = new NotificationsController();

                $newTransaction = transactions::where('transaction_id', '=', $request->transactionId)->first();
                $state = "Update";
                //   $transactionIndex =  $historyController->insertTransactionHistoryDataByRef( 'Update',$newTransaction ->transaction_reference,auth()->user()->id,$date);

                //       $transactionIndex = $historyController->insertTransactionHistoryData('transactions', $transaction, 'Update', $hitoryArray);
            }

            if (!$checkTransactionTmpExistance->transactiontmp_reference) {
                sequence::where('sequence_transactionType', 'like', $checkTransactionTmpExistance->transactiontmp_type)->increment('sequence_transactionRef', 1);
            }

            // after saving and creation of transaction or update get tmp transaction dt into original and remove from tmp
            if ($newTransaction) {
                // start : transfer supplier to client suppliers
                if ($newTransaction->transaction_supplierId) {
                    $this->createSupplier($newTransaction);
                }

                $insertDtFound = array();
                foreach ($transactionDtsTmp as $transactionDtTmp) {
                    if ($transactionDtTmp->state == 'deleted') continue;

                    $newTransactionDt = new transactionDt();
                    $newTransactionDt->transactiondt_id = $transactionDtTmp->transactiondttmp_id;
                    $newTransactionDt->transaction_id = $newTransaction->transaction_id;
                    $newTransactionDt->item_id = $transactionDtTmp->item_id;
                    $newTransactionDt->client_id = $transactionDtTmp->client_id;
                    $newTransactionDt->item_quantity = $transactionDtTmp->item_quantity;
                    $newTransactionDt->supplier_id = $checkTransactionTmpExistance->transactiontmp_supplierId;
                    $newTransactionDt->supplier_invoice_number = $checkTransactionTmpExistance->transactiontmp_supplierInvoice;
                    $newTransactionDt->transactiondt_locationid = $transactionDtTmp->transactiondttmp_locationid;
                    $newTransactionDt->transactiondt_countryOfOrigin = $transactionDtTmp->transactiondttmp_countryOfOrigin;
                    $newTransactionDt->transactiondt_price = $transactionDtTmp->transactiondttmp_price;
                    $newTransactionDt->transactiondt_priceBase = $transactionDtTmp->transactiondttmp_priceBase;
                    $newTransactionDt->id_reference_dt = $transactionDtTmp->id_reference_dt;
                    $newTransactionDt->transactiondt_billOfEntry = $checkTransactionTmpExistance->transactiontmp_billOfEntry;
                    $newTransactionDt->raw_in_grid = $transactionDtTmp->raw_in_grid;
                    $newTransactionDt->item_weight = $transactionDtTmp->item_weight;
                    $newTransactionDt->item_unit = $transactionDtTmp->item_unit;
                    $newTransactionDt->item_qtyPerPack = $transactionDtTmp->item_qtyPerPack;
                    $newTransactionDt->item_pack = $transactionDtTmp->item_pack;
                    $newTransactionDt->group_id = $transactionDtTmp->group_id;
                    $newTransactionDt->palet_no = $transactionDtTmp->palet_no;
                    $newTransactionDt->transactiondt_nbOfCarton = $transactionDtTmp->transactiondttmp_nbOfCarton;
                    $newTransactionDt->transactiondt_cellMerge = $transactionDtTmp->transactiondttmp_cellMerge;
                    $newTransactionDt->transactiondt_BOE = $transactionDtTmp->transactiontmpdt_BOE;
                    $newTransactionDt->transactiondt_fprice = $transactionDtTmp->transactiondttmp_fprice;
                    $newTransactionDt->transactiondt_fpriceBase = $transactionDtTmp->transactiondttmp_fpriceBase;
                    $newTransactionDt->hscode = $transactionDtTmp->hscode;
                    $newTransactionDt->transactiondt_item_description = $transactionDtTmp->transactiondt_item_description;
                    $newTransactionDt->total_price = $transactionDtTmp->total_price;
                    $newTransactionDt->total_weight = $transactionDtTmp->total_weight;
                    $newTransactionDt->total_gross_weight = $transactionDtTmp->total_gross_weight;
                    $newTransactionDt->gross_weight = $transactionDtTmp->gross_weight;


                    $newTransactionDt->save();


                    array_push($insertDtFound, $transactionDtTmp->transactiondttmp_id);
                    // start : transfer items to client items

                    if ($transactionDtTmp->item_id) {
                        $this->createClientItems($transactionDtTmp, $newTransaction->transaction_clientId);
                    }
                    // end : transfer items to client items


                    //   $newTransactionDt = transactiondt::create($transactionDtArray);


                    $previousSnData = null;
                    $serialsTmp = transactionDtSnTmp::where('transactiondttmp_id', '=', $transactionDtTmp->transactiondttmp_id)->get();
                    $previousSnData = transactionDtsn::where('transactiondt_id', $transactionDtTmp->transactiondttmp_id)->get();

                    transactionDtsn::where('transactiondt_id', '=', $transactionDtTmp->transactiondttmp_id)->delete();
                    $insertSnFound = array();
                    foreach ($serialsTmp as $serialTmp) {

                        $transactionSerialTmpArray = array(
                            'transactiondt_id' => $newTransactionDt->transactiondt_id,
                            'transactiondtsn_id' => $serialTmp->transactiondtsntmp_id,
                            'item_id' => $serialTmp->item_id,
                            'transaction_item_serial' => $serialTmp->transaction_item_serial
                        );
                        if ($serialTmp->status == 'Delete') {


                            continue;
                        }
                        $newTransactionDtsn = transactionDtsn::create($transactionSerialTmpArray);
                    }
                }
                $historyController->insertTransactionHistoryDataByRef(
                    $state,
                    $newTransaction->transaction_reference,
                    auth()->user()->id,
                    $date,
                    $newTransaction->transaction_type
                );
                DB::table('client_storage_pallets')
                    ->where('related_transaction_id', '=', $request->transactionTmpId)
                    ->delete();
                $storageData = DB::table('client_storage_pallets_tmp')
                    ->where('related_transaction_id', $request->transactionTmpId)
                    ->get();
                foreach ($storageData as $item) {
                    $data = array(
                        'id' => $item->id,
                        'length' => $item->length,
                        'width' => $item->width,
                        'height' => $item->height,
                        'related_transaction_id' => $item->related_transaction_id,
                        'pallet_no' => $item->pallet_no,
                        'pallet_seq' => $item->pallet_seq,
                        'related_plan' => $item->related_plan,
                        'date' => $checkTransactionTmpExistance->transactiontmp_date,
                        'transaction_type' => $checkTransactionTmpExistance->transactiontmp_type,

                    );
                    DB::table('client_storage_pallets')
                        ->insert($data); // Replace property_name with the actual property you want to access
                }
                DB::table('client_storage_pallets_tmp')
                    ->where('related_transaction_id', '=', $request->transactionTmpId)
                    ->delete();
                transactiontmp::where('transactiontmp_id', '=', $request->transactionTmpId)->delete();
            }
            //  $historyController->insertHistoryTransactionDtData($previousDtData, $insertDtFound, $date, $transactionIndex, auth()->user()->id);
        }
        return ApiController::errorResponse($invalidTransactionsIndex, 200);
    }

    public function createSupplier($newTransaction)
    {
        $getSupplierDetails = suppliers::where('supplier_id', '=', $newTransaction->transaction_supplierId)->first();
        $checkIfSupplierExist = clientSupplier::where([
            ['supplier_id', '=', $getSupplierDetails->supplier_id],
            ['supplier_related_clientid', '=', $newTransaction->transaction_clientId]
        ])
            ->first();

        if (!$checkIfSupplierExist && $getSupplierDetails) {
            $supplierArray = array(
                'supplier_id' => $getSupplierDetails->supplier_id,
                'supplier_name' => $getSupplierDetails->supplier_name,
                'supplier_contactPerson' => $getSupplierDetails->supplier_contactPerson,
                'supplier_address' => $getSupplierDetails->supplier_address,
                'supplier_country' => $getSupplierDetails->supplier_country,
                'supplier_phone' => $getSupplierDetails->supplier_phone,
                'supplier_email' => $getSupplierDetails->supplier_email,
                'supplier_related_clientid' => $newTransaction->transaction_clientId
            );


            clientSupplier::create($supplierArray);
        }
    }

    public function fetchTransactions(Request $request)
    {
        $utilitiesController = new UtilitiesController();
        $count = 0;

        $transactions = transactions::leftjoin('clients', 'clients.client_id', '=', 'transactions.transaction_clientId')
            ->leftjoin('suppliers', 'suppliers.supplier_id', '=', 'transactions.transaction_supplierId')
            ->leftjoin('client_customers', 'client_customers.customer_id', '=', 'transactions.clientCustomer_id')
            ->leftjoin('transaction_dts', 'transaction_dts.transaction_id', '=', 'transactions.transaction_id')
            ->select(
                'customer_name',
                'transactions.transaction_reference',
                'transactions.transaction_date',
                'clients.client_firstname',
                'transactions.transaction_id',
                'transactions.transaction_mobileReference',
                'salesjv_id',
                'transactions.transaction_remark',
                'transactions.transaction_billOfEntry',
                'transactions.transaction_shipmentReference',
                'transactions.transaction_typeOfDeliveryId',
                'suppliers.supplier_name',
                'transactions.transaction_supplierInvoice',
                'transactions.transaction_type',
                'transaction_BOE',
                'Invoice_type_id',
                'payment_term_id',
                'delivery_term_id',
                'exemption_type_id',
                'transaction_dts.transactiondt_id',

            )
            //->selectRaw(DB::raw(" FROM_UNIXTIME(transactions.transaction_date, '%d-%m-%Y') AS transaction_formatdate"))
            ->where([
                ['transactions.transaction_type', $request->transactionType],
                ['internal_transfer', transactions::transactionTypes["$request->transactionType"]["internalTransfer"]]

            ])
            ->whereRaw('date(FROM_UNIXTIME(transaction_date)) BETWEEN ? AND ?', [$request->dateFrom, $request->dateTo])
            ->groupby('transactions.transaction_id')
            ->orderBy('transactions.transaction_date', 'desc')
            ->orderBy('transactions.transaction_reference', 'desc');
        $count = count($transactions->get());
        if (isset($request->skip) && isset($request->take)) {
            $transactions->skip($request->skip)
                ->take($request->take);
        }
        if ($request->search) {
            $transactions->Where(function ($query) use ($request) {
                $query->Where([
                    ['transaction_reference', 'like', '%' . $request->search . '%']
                ])
                    ->orWhere([
                        ['transaction_billOfEntry', 'like', '%' . $request->search . '%']
                    ])
                    ->orWhere([
                        ['transaction_supplierInvoice', 'like', '%' . $request->search . '%']
                    ])
                    ->orWhere([
                        ['transaction_BOE', 'like', '%' . $request->search . '%']
                    ])
                    ->orWhere([
                        ['clients.client_firstname', 'like', '%' . $request->search . '%']
                    ]);
            });
        }

        $transactions = $transactions->get();
        foreach ($transactions as $splitTrans) {
            $splitTrans->transaction_formatdate = date('Y-m-d', $splitTrans->transaction_date);

            $totalRemain = transactions::remainingQty($splitTrans->transactiondt_id);

            $splitTrans->DOTmpBILL = null;
            if ($totalRemain == 0) {
                $splitTrans->DOTmpBILL = 1;
            }
        }



        if ($transactions) {
            $commonServices = new CommonServices;
            $transactions = $commonServices->getTransactionFilesExist(['GOODS_IN'], $transactions);

            return ApiController::successResponse($transactions, 200, $count);
        }
        return ApiController::successResponse('No result', 204);
    }

    public function insertNewTransactionSave(Request $request)
    {
        $date = date('Y-m-d H:i:s');
        $transactionIndex = 0;

        // retrieve the current transaction tmp
        $checkTransactionTmpExistance = transactiontmp::where('transactiontmp_id', '=', $request->transactionTmpId)
            ->first();

        // retrieve the current transaction dt tmp
        $transactionDtsTmp = transactiondttmp::join('items', 'items.item_id', '=', 'transactiondttmps.item_id')
            ->select('transactiondttmps.*', 'items.item_hasSerial')
            ->where('transactiontmp_id', '=', $request->transactionTmpId)
            ->get();


        $invalidTransactionsIndex = array();
        $navigationTypeOUT = $checkTransactionTmpExistance->transactiontmp_type == 'GOODS_IN' ? 'DO' : 'RMA_OUT';

        $transactionDtsCount = count($transactionDtsTmp); // Cache count() value for better performance

        DB::beginTransaction();

        try {
            // Insert transactiondt
            for ($i = 0; $i < $transactionDtsCount; $i++) {

                $transactiondtTmp = $transactionDtsTmp[$i];

                $transactiondt = new transactionDt();
                $transactiondt->transaction_id = 2;
                $transactiondt->transactiondt_id = $transactiondtTmp->transactiondttmp_id;
                $transactiondt->transactiondt_billOfEntry = $transactiondtTmp->transactiontmpdt_billOfEntry;
                $transactiondt->transactiondt_countryOfOrigin = $transactiondtTmp->transactiondttmp_countryOfOrigin;
                $transactiondt->item_id = $transactiondtTmp->item_id;
                $transactiondt->item_quantity = $transactiondtTmp->item_quantity;
                $transactiondt->created_at = $date;
                $transactiondt->updated_at = $date;

                $transactiondt->save();

                // Insert transactiondt_sn
                if ($transactiondtTmp->item_hasSerial == 'YES') {
                    $serials = transactionDtSnTmp::where('transactiondttmp_id', '=', $transactiondtTmp->transactiondttmp_id)
                        ->where(function ($query) {
                            $query->where('status', '!=', 'Delete')
                                ->orWhereNull('status');
                        })
                        ->get();

                    foreach ($serials as $serial) {
                        $transactiondtSn = new transactionDtsn();
                        $transactiondtSn->transactiondt_id = $transactiondt->transactiondt_id;
                        $transactiondtSn->serial_number = $serial->serial_number;
                        $transactiondtSn->status = 'Open';
                        $transactiondtSn->created_at = $date;
                        $transactiondtSn->updated_at = $date;
                        $transactiondtSn->save();
                    }
                }

                // Update transactiondttmp
                $transactiondtTmp->transaction_id = $transactiondt->transaction_id;
                $transactiondtTmp->save();
            }

            // Update transactiontmp
            $checkTransactionTmpExistance->transactiontmp_index = $transactionIndex;
            $checkTransactionTmpExistance->save();

            DB::commit();
            return response()->json(['success' => true, 'message' => 'Transaction inserted successfully']);
        } catch (Exception $e) {

            // DB::rollback();

            return response()->json(['success' => false, 'message' => 'Failed to insert transaction', 'error' => $e->getMessage()]);
        }
    }

    public function updateTransaction(Request $request)
    {
        $utilitiesController = new UtilitiesController();
        $validator = Validator::make($request->all(), [
            'transactionDate' => 'required',
            'transactionClient' => 'required',
            'transactionBillOfEntry' => 'required'
            //'transactionRemark' => 'required'
        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 422);
        }

        $transaction = array(
            //'transaction_reference' => $sequenceNumber,
            'transaction_date' => $utilitiesController->convertTodayToTimetsamp($request->transactionDate . "" . date('H:i:s')),
            //'transaction_clientId' => $request->transactionClient,
            //'transaction_type' => 'GOODS_IN',
            'transaction_billOfEntry' => $request->transactionBillOfEntry,
            'transaction_remark' => $request->transactionRemark,
            'transaction_shipmentReference' => $request->transactionShipmentReference,
        );

        $updateTransaction = transactions::where('transaction_id', $request->transactionId)
            ->update($transaction);

        if ($request->transactionDetails) {
            $transactionDetails = $request->transactionDetails;
            for ($i = 0; $i < count($transactionDetails); $i++) {
                $element = $transactionDetails[$i];
                if ($element['transactionDetailItemCode'] || $element['transactionDetailItemBarcode']) {
                    $transactionDt = array(
                        'item_id' => $element['transactionDetailItemCode'],
                        'item_quantity' => $element['transactionDetailItemQty'],
                        'supplier_id' => $element['transactionDetailSupplier'],
                        'supplier_invoice_number' => $element['transactionDetailSupplierInvoiceNumber'],
                        'transactiondt_locationid' => $element['transactionDetailLocation'],
                        'transactiondt_countryOfOrigin' => $element['transactionDetailCountryOfOrigin'],
                        'transactiondt_price' => $element['transactionDetailPrice'],


                        'transactiondt_priceBase' => isset($element['transactionDetailPriceBase']) ? $element['transactionDetailPriceBase'] : $element['transactionDetailPrice'],
                        'id_reference_dt' => $element['transactionDetailIdReferenceDt'],
                        'raw_in_grid' => $element['transactionDetailRaw']
                    );
                    if ($element['transactiondt_id'] != null) {
                        transactionDt::where('transactiondt_id', '=', $element['transactiondt_id'])->update($transactionDt);
                    } else {
                        $transactionDt['transaction_id'] = $request->transactionId;
                        //$transactionDt['transactiondt_locationid'] = 1;
                        transactionDt::create($transactionDt);
                    }
                }
            }
        }

        if ($updateTransaction) {
            return ApiController::successResponse($updateTransaction, 200);
        }
        return ApiController::errorResponse($updateTransaction, 422);
    }

    public function fetchPendingTransactions(Request $request)
    {
        $loggerUser = auth()->user()->id;
        $useraccess = DB::table('useraccess')
            ->where([
                ['user_id', '=', $loggerUser],
                ['useraccess_formcode', 'like', 'view-All-Pending']
            ])
            ->first();

        $transactions = transactiontmp::join('users', 'users.id', '=', 'transactiontmps.user_id')
            ->join('clients', 'clients.client_id', '=', 'transactiontmps.transactiontmp_clientId')
            ->leftjoin('client_customers', 'client_customers.customer_id', '=', 'transactiontmps.clientCustomer_id')
            ->leftjoin('client_forwarders', 'client_forwarders.clientforwarder_id', '=', 'transactiontmps.forwarder_id')
            ->orderBy('transactiontmp_date', 'desc')
            ->where([
                ['transactiontmps.transactiontmp_type', transactions::transactionTypes["$request->transactionType"]["type"]],
                ['transactiontmps.internal_Transfer', transactions::transactionTypes["$request->transactionType"]["internalTransfer"]]
            ]);

        if (!$useraccess || ($useraccess && !$useraccess->useraccess_view)) {
            $transactions->where([
                ['transactiontmps.user_id', '=', $loggerUser]
            ]);
        }
        $transactions = $transactions->get();

        foreach ($transactions as $transaction) {

            if ($transaction->transactiontmp_date) {
                $transaction->transactiontmp_formatdate = date('Y-m-d', $transaction->transactiontmp_date);
            }
        }
        return ApiController::successResponse($transactions, 200);
    }

    public function fetchMobilePendingTransactions(Request $request)
    {
        $loggerUser = auth()->user()->id;

        $transactions = transactiontmp::leftjoin('users', 'users.id', '=', 'transactiontmps.user_id')
            ->join('clients', 'clients.client_id', '=', 'transactiontmps.transactiontmp_clientId')
            ->leftjoin('client_customers', 'client_customers.customer_id', '=', 'transactiontmps.clientCustomer_id')
            ->leftjoin('client_forwarders', 'client_forwarders.clientforwarder_id', '=', 'transactiontmps.forwarder_id')
            ->orderBy('transactiontmp_date', 'desc')
            ->where([
                ['transactiontmps.transactiontmp_type', 'like', $request->transactionType],
                ['transactiontmps.user_id', '=', null],
            ])
            ->get();
        foreach ($transactions as $transaction) {
            if ($transaction->transactiontmp_date) {
                $transaction->transactiontmp_formatdate = date('Y-m-d', $transaction->transactiontmp_date);
            }
        }
        return ApiController::successResponse($transactions, 200);
    }

    public function updateUserIdIntransactionTmp(Request $request)
    {
        $loggerUser = auth()->user()->id;

        $updateTransactionTmp = transactiontmp::where('transactiontmp_id', $request->transactionTmpId)
            ->update(['user_id' => $loggerUser]);

        return ApiController::successResponse($updateTransactionTmp, 200);
    }

    public function fetchTransactionDetails(Request $request)
    {
        return $this->fetchTransactionDetailsFunction($request->transactionId);
    }

    public function fetchTransactionDetailsFunction($transactionId)
    {
        $transactionDetails = transactionDt::join('items', 'items.item_id', '=', 'transaction_dts.item_id')
            ->leftjoin('client_suppliers', 'client_suppliers.supplier_id', '=', 'transaction_dts.supplier_id')
            ->select(
                'items.item_barcode',
                'client_suppliers.supplier_name',
                'transaction_dts.item_quantity',
                'transaction_dts.supplier_invoice_number',
                'transaction_dts.item_id',
                'transaction_dts.supplier_id',
                'transaction_id',
                'transactiondt_id',
                'items.item_code',
                'items.item_description',
                'transaction_dts.transactiondt_locationid',
                'transaction_dts.transactiondt_countryOfOrigin',
                'transaction_dts.group_id',
                'transaction_dts.palet_no',
                'items.item_hasSerial',
                'transaction_dts.transactiondt_price',
                'transaction_dts.id_reference_dt',
                'transaction_dts.raw_in_grid',
                'transaction_dts.transactiondt_billOfEntry',
                'transaction_dts.total_gross_weight',
                'transaction_dts.gross_weight',


            )
            ->where([
                ['transaction_id', '=', $transactionId],
                //  ['state','!=','deleted']
            ])
            ->get();

        $getClientTransaction = transactions::where([
            ['transactions.transaction_id', '=', $transactionId]
        ])
            ->select('transactions.transaction_clientId')
            ->first();

        if ($getClientTransaction && $getClientTransaction->transaction_clientId) {
            $getClientItems = clientItem::where([
                ['item_related_client_id', '=', $getClientTransaction->transaction_clientId]
            ])
                ->get();

            $getClientSuppliers = clientSupplier::where([
                ['client_suppliers.supplier_related_clientid', '=', $getClientTransaction->transaction_clientId]
            ])
                ->get();
        }

        $data = array();
        $data['clientItems'] = $getClientTransaction ? $getClientItems : null;
        $data['clientSuppliers'] = $getClientTransaction ? $getClientSuppliers : null;
        $data['locations'] = location::get();

        foreach ($transactionDetails as $transactionDt) {
            $transactionDt->transactionDtSerials = $this->fetchTransactionDtSerials($transactionDt->transactiondt_id);
        }
        $data['transactionDetails'] = $transactionDetails;

        return ApiController::successResponse($data, 200);
    }

    public function fetchTransactionDtSerials($transactiondtId)
    {
        $transactionSerials = transactionDtsn::where('transactiondt_id', $transactiondtId)
            ->get();

        $getTransactionClient =
            transactionDt::join('transactions', 'transactions.transaction_id', '=', 'transaction_dts.transaction_id')
            ->where('transactiondt_id', $transactiondtId)
            ->select('transactions.transaction_clientId')
            ->first();

        if ($getTransactionClient) {
            $getClientItems = clientItem::where([
                ['item_related_client_id', '=', $getTransactionClient->transaction_clientId]
            ])
                ->get();
        }
        // $data = Array();
        // $data['transactionSerials'] = $transactionSerials;
        //$data['clientItems'] = $getTransactionClient ? $getClientItems : null;

        return $transactionSerials;
    }

    public function fetchTransactionDetailsTmp(Request $request)
    {
        $transactionDetailsTmp = transactiondttmp::join('items', 'items.item_id', '=', 'transactiondttmps.item_id')
            ->leftjoin('suppliers', 'suppliers.supplier_id', '=', 'transactiondttmps.supplier_id')
            ->leftjoin('locations', 'locations.location_id', '=', 'transactiondttmps.transactiondttmp_locationid')
            ->leftjoin('countries', 'countries.country_id', '=', 'transactiondttmps.transactiondttmp_countryOfOrigin')
            ->select(
                'items.item_barcode',
                'suppliers.supplier_name',
                'transactiondttmps.item_quantity',
                'transactiondttmps.supplier_invoice_number',
                'transactiondttmps.group_id',
                'transactiondttmps.palet_no',
                'transactiondttmps.item_id',
                'transactiondttmps.supplier_id',
                'transactiondttmps.item_pack',
                'transactiondttmps.item_qtyPerPack',
                'transactiondttmps.transactiondttmp_nbOfCarton',

                'transactiontmp_id',
                'transactiondttmp_id',
                'items.item_code',
                'items.item_description',
                'transactiondttmps.transactiontmpdt_billOfEntry',
                'transactiondttmps.transactiondttmp_locationid',
                'transactiondttmps.transactiondttmp_countryOfOrigin',
                'items.item_hasSerial',
                'country_name',
                'transactiondttmps.item_weight',
                'transactiondttmps.transactiondttmp_price',
                'transactiondttmps.transactiondttmp_priceBase',
                'transactiondttmps.id_reference_dt',
                'transactiondttmps.raw_in_grid',
                'locations.location_id',
                'locations.location_name',
                'transactiondttmp_fprice',
                'hscode',
                'transactiondt_item_description',
                'transactiondttmps.total_price',
                'transactiondttmps.total_weight',
                'transactiondttmps.total_gross_weight',
                'transactiondttmps.gross_weight',


            )
            ->where([
                ['transactiontmp_id', '=', $request->transactionTmpId],
                ['state', '!=', 'deleted']
            ])
            ->get();


        $loggerUserCompany = auth()->user()->company_id;

        if ($loggerUserCompany) {
            $getClientItems = items::where([
                ['company_id', '=', $loggerUserCompany]
            ])
                ->get();

            $getClientSuppliers = suppliers::where([
                ['company_id', '=', $loggerUserCompany]
            ])
                ->get();
        }

        $data = array();
        $data['clientItems'] = $loggerUserCompany ? $getClientItems : null;
        $data['clientSuppliers'] = $loggerUserCompany ? $getClientSuppliers : null;
        $data['locations'] = location::get();

        $data['transactionDetailsTmp'] = $transactionDetailsTmp;


        return ApiController::successResponse($data, 200);
    }

    public function fetchTransactionDetailsDO(Request $request)
    {
        if ($request->rawInGrid != null) {
            $transactionDetails = transactiondt::join('client_items', 'client_items.item_id', '=', 'transaction_dts.item_id')
                ->leftjoin('client_suppliers', 'client_suppliers.supplier_id', '=', 'transaction_dts.supplier_id')
                ->select(
                    'client_items.item_barcode',
                    'client_suppliers.supplier_name',
                    'transaction_dts.item_quantity',
                    'transaction_dts.supplier_invoice_number',
                    'transaction_dts.item_id',
                    'transaction_dts.supplier_id',
                    'transaction_id',
                    'transactiondt_id',
                    'client_items.item_code',
                    'client_items.item_description',
                    'transaction_dts.transactiondt_billOfEntry',
                    'transaction_dts.transactiondt_locationid',
                    'transaction_dts.transactiondt_countryOfOrigin',
                    'transaction_dts.transactiondt_price',
                    'transaction_dts.id_reference_dt',
                    'transaction_dts.raw_in_grid',
                    'client_items.item_hasSerial',
                )
                ->where([
                    ['transaction_id', '=', $request->transactionId],
                    ['raw_in_grid', $request->rawInGrid]
                ])
                ->get();
            $getClientTransaction = transactions::where([
                ['transactions.transaction_id', '=', $request->transactionId]
            ])
                ->select('transactions.transaction_clientId')
                ->first();

            //return $getClientTransaction."";

            if ($getClientTransaction && $getClientTransaction->transaction_clientId) {
                $getClientItems = clientItem::where([
                    ['item_related_client_id', '=', $getClientTransaction->transaction_clientId]
                ])
                    ->get();

                $getClientSuppliers = clientSupplier::where([
                    ['client_suppliers.supplier_related_clientid', '=', $getClientTransaction->transaction_clientId]
                ])
                    ->get();
            }
            $data = array();
            $data['transactionDetails'] = $transactionDetails;
            foreach ($data['transactionDetails'] as $transactionDt) {
                $transactionDt->transactionDtSerials = $this->fetchTransactionDtSerials($transactionDt->transactiondt_id);
            }
            return ApiController::successResponse($data, 200);
        } else {
            $transactionDetailsTmp = DB::table('transaction_dts')
                ->join('client_items', 'client_items.item_id', '=', 'transaction_dts.item_id')
                ->select('transaction_dts.raw_in_grid', 'transaction_dts.transactiondt_price', 'transaction_dts.transactiondt_priceBase', 'client_items.item_description', 'client_items.item_id', 'client_items.item_code')
                ->selectRaw(' SUM(transaction_dts.item_quantity) as item_quantity')
                ->groupBy('transaction_dts.raw_in_grid', 'transaction_dts.transactiondt_price', 'transaction_dts.transactiondt_priceBase', 'client_items.item_description', 'client_items.item_id', 'client_items.item_code')
                ->where([
                    ['transaction_dts.transaction_id', '=', $request->transactionId]
                ])
                ->get();

            return ApiController::successResponse($transactionDetailsTmp, 200);
        }
    }


    //out in serials

    public function fetchClientDetails(Request $request)
    {
        $getClientItems = clientItem::where([
            ['item_related_client_id', '=', $request->clientId]
        ])
            ->get();

        $getClientSuppliers = clientSupplier::where([
            ['client_suppliers.supplier_related_clientid', '=', $request->clientId]
        ])
            ->get();

        $data = array();
        $data['clientItems'] = $getClientItems;
        $data['clientSuppliers'] = $getClientSuppliers;
        $data['locations'] = location::get();

        return ApiController::successResponse($data, 200);
    }

    public function insertTransactionDtSerials(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'transactionSerials' => 'required|array',
            'itemId' => 'required'
        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 422);
        }

        for ($i = 0; $i < count($request->transactionSerials); $i++) {
            $element = $request->transactionSerials[$i];
            if ($request->itemId && $element['transactionDetailSerialNumber']) {
                $transactionSerial = array(
                    'item_id' => $request->itemId,
                    'transaction_item_serial' => $element['transactionDetailSerialNumber'],
                    'transactiondt_id' => $request->transactionDetailId
                );
                if ($element['transactiondtsn_id']) {
                    transactionDtsn::where('transactiondtsn_id', $element['transactiondtsn_id'])
                        ->update($transactionSerial);
                } else {
                    transactionDtsn::create($transactionSerial);
                }
            }
        }
        return ApiController::successResponse($request->transactionSerials, 200);
    }

    public function insertTransactionDtSerialTmp(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'transactionSerial' => 'required|array',
            'transactionDtTmp' => 'required'
        ]);
        $loggerUserCompany = auth()->user()->company_id;

        if ($validator->fails()) {
            return response()->json($validator->errors(), 422);
        }

        $invalidSerials = array();
        for ($i = 0; $i < count($request->transactionSerial); $i++) {
            $element = $request->transactionSerial[$i];
            $result = '';
            $result = implode(';', $element);

            $arrayOfSerials = explode(';', $result);
            $getItem = Items::where('item_code', trim($request->transactionDtTmp['item_id']))
                ->first();
            $noOfFieldsPerItem = DB::table('serial_types')
                ->join('items', 'items.serial_type_id', 'serial_types.serial_id')
                ->where('item_id', $getItem->item_id)
                ->count();

            if (count($arrayOfSerials) != $noOfFieldsPerItem) {
                array_push($invalidSerials, $result);

                continue;
            }


            $transactionSerialArray = array(
                'item_id' => $getItem->item_id,
                'status' => 'New',
                'transaction_item_serial' => $result,
                'transactiondttmp_id' => $request->transactionDtTmp['transactiondttmp_id'],
            );

            $checkSerialExistanceTmp = null;
            $checkSerialExistance = null;
            if ($request->transactionDtTmp) {

                $checkSerialExistanceTmp = transactionDtSnTmp::join('transactiondttmps', 'transactiondttmps.transactiondttmp_id', 'transaction_dt_sn_tmps.transactiondttmp_id')
                    ->join('transactiontmps', 'transactiontmps.transactiontmp_id', 'transactiondttmps.transactiontmp_id')
                    ->where([
                        ['transactiontmps.company_id', $loggerUserCompany],
                    ])
                    ->where(function ($query) use ($arrayOfSerials) {
                        foreach ($arrayOfSerials as $serial) {
                            $query->orWhereRaw("FIND_IN_SET('$serial', REPLACE(transaction_item_serial, ';', ',')) > 0");
                        }
                    })
                    ->where(function ($query) {
                        $query->where('status', '!=', 'Delete')
                            ->orWhereNull('status');
                    })
                    ->first();


                $checkSerialExistance = transactionDtSn::join('transaction_dts', 'transaction_dts.transactiondt_id', 'transaction_dtsns.transactiondt_id')
                    ->join('transactions', 'transactions.transaction_id', 'transaction_dts.transaction_id')
                    ->where([

                        ['transactions.company_id', $loggerUserCompany],
                        ['transaction_dts.transactiondt_id', '!=', $request->transactionDtTmp['transactiondttmp_id']]
                    ])
                    ->where(function ($query) use ($arrayOfSerials) {
                        foreach ($arrayOfSerials as $serial) {
                            $query->orWhereRaw("FIND_IN_SET('$serial', REPLACE(transaction_item_serial, ';', ',')) > 0");
                        }
                    })
                    ->first();
            }
            $isValid = $this->checkSerialifOut($loggerUserCompany, $request, $arrayOfSerials);
            if (!$checkSerialExistanceTmp && $isValid == 1) {

                $transactionSerialArray['status'] = 'Update';
                if (!$request->transactionDtSerialTmp) {
                    transactionDtSnTmp::create($transactionSerialArray);
                } else {

                    transactionDtSnTmp::where([
                        ['transactiondtsntmp_id', '=', $request->transactionDtSerialTmp['transactiondtsntmp_id']],

                    ])
                        ->where(function ($query) {
                            $query->where('status', '!=', 'Delete')
                                ->orWhereNull('status');
                        })
                        ->update($transactionSerialArray);
                }
            }
        }

        $request['transactiondttmp_id'] = $request->transactionDtTmp['transactiondttmp_id'];

        $transactionDtSerials = $this->fetchSerialsTmp($request)->getData();
        if (count($invalidSerials) > 0) {
            return ApiController::errorResponse($transactionDtSerials, 422);
        }
        return $transactionDtSerials;
    }

    public function checkSerialifOut($loggerUserCompany, $request, $arrayOfSerials)
    {

        $counts = transactionDtSn::join('transaction_dts', 'transaction_dts.transactiondt_id', 'transaction_dtsns.transactiondt_id')
            ->join('transactions', 'transactions.transaction_id', 'transaction_dts.transaction_id')
            ->where([
                ['transactions.company_id', $loggerUserCompany],
                ['transaction_dts.transactiondt_id', '!=', $request->transactionDtTmp['transactiondttmp_id']]
            ])
            ->where(function ($query) use ($arrayOfSerials) {
                foreach ($arrayOfSerials as $serial) {
                    $query->orWhereRaw("FIND_IN_SET('$serial', REPLACE(transaction_item_serial, ';', ',')) > 0");
                }
            })
            ->selectRaw('transactions.transaction_type, COUNT(*) as count')
            ->groupBy('transactions.transaction_type')
            ->get();

        $sumGoodsIn = $counts->where('transaction_type', 'GOODS_IN')->sum('count');
        $sumDO = $counts->where('transaction_type', 'DO')->sum('count');

        $total = $sumGoodsIn - $sumDO;
        if ($total == 0) {
            return 1;
        }
        return 0;
    }

    public function fetchSerialsTmp(Request $request)
    {
        $transactionSerials = DB::table('transaction_dt_sn_tmps')
            ->leftjoin('transactiondttmps', 'transactiondttmps.transactiondttmp_id', 'transaction_dt_sn_tmps.transactiondttmp_id')
            ->select(
                'transaction_dt_sn_tmps.transactiondtsntmp_id',
                'transaction_dt_sn_tmps.transaction_item_serial',
                'transaction_dt_sn_tmps.transactiondttmp_id',
                'transactiondttmps.item_id',
                'transactiondttmps.transactiondttmp_id'
            )
            ->where([
                ['transactiondttmps.transactiondttmp_id', $request->transactiondttmp_id],
            ])
            ->where(function ($query) {
                $query->where('status', '!=', 'Delete')
                    ->orWhereNull('status');
            })
            ->get();

        $serialsHeadear = Db::table('transactiondttmps')
            ->leftjoin('items', 'items.item_id', 'transactiondttmps.item_id')
            ->leftjoin('serial_types', 'serial_types.serial_id', 'items.serial_type_id')
            ->where([
                ['transactiondttmps.transactiondttmp_id', $request->transactiondttmp_id],
            ])
            ->select('serial_types.serial_label')
            ->pluck('serial_types.serial_label');

        foreach ($transactionSerials as $serial) {
            $splitSerials = explode(';', $serial->transaction_item_serial);
            for ($i = 0; $i < count($serialsHeadear); $i++) {
                $header = $serialsHeadear[$i];
                $serial->{$header} = isset($splitSerials[$i]) ? $splitSerials[$i] : "-";
            }
        }

        $data["header"] = $serialsHeadear;
        $data["transactionSerials"] = $transactionSerials;
        return ApiController::successResponse($data, 200);

        $processedSerials = [];
        foreach ($transactionSerials as $serial) {
            $splitSerials = explode(';', $serial->transaction_item_serial);
            $serial->transaction_item_serial = $splitSerials;

            $processedSerials[] = $serial;
        }

        $processedSerialsJson = json_encode($processedSerials);

        return ApiController::successResponse($processedSerialsJson, 200);
    }

    public function updateTransactionDtSerialsItem(Request $request)
    {
        $validator = Validator::make($request->all(), [
            //'transactionDtId' => 'required',
            'itemId' => 'required'
        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 422);
        }

        $updateTransactionSerials = transactionDtsn::where('transactiondt_id', '=', $request->transactionDtId)
            ->update([
                'item_id' => $request->itemId
            ]);

        return ApiController::successResponse($updateTransactionSerials, 200);
    }

    public function fetchTransactionDtSerialsTmp($transactionDtTmpId)
    {
        $transactionSerials = transactionDtSnTmp::where([
            ['transactiondttmp_id', $transactionDtTmpId],
            ['status', '!=', 'Delete']
        ])
            ->get();

        return $transactionSerials;
    }

    public function cancelTransactionTmpTable(Request $request)
    {
        $TransationDtTmp = transactiondttmp::where('transactiontmp_id', '=', $request->transactionTmpId)->get();
        $transactionType = transactiontmp::where('transactiontmp_id', '=', $request->transactionTmpId)->first();
        $cancelTransactionTmp = transactiontmp::where('transactiontmp_id', '=', $request->transactionTmpId)->delete();
        $cancelTransactiontmpPalets = TransactiontmpPalets::where('transactiontmp_id', '=', $request->transactionTmpId)->delete();

        if (!$transactionType) {
            return ApiController::successResponse($cancelTransactionTmp, 200);
        }
        foreach ($TransationDtTmp as $transactionDetailTmp) {
            transactionDtSnTmp::where('transactiondttmp_id', '=', $transactionDetailTmp->transactiondttmp_id)->delete();
        }
        $cancelTransationDtTmp = transactiondttmp::where('transactiontmp_id', '=', $request->transactionTmpId)
            ->delete();
        DB::table('client_storage_pallets_tmp')->where('related_transaction_id', $request->transactionTmpId)->delete();

        $path = public_path() . '/documents/' . $transactionType->transactiontmp_type . '/' . $request->transactionTmpId . '/' . $request->file;

        if (file_exists($path) && $transactionType->transactiontmp_reference == null) {
            $deleteFile = File::deleteDirectory($path);
        }


        $transferTrans = transactiontmp::where('relatedTransferId', $request->transactionTmpId)
            ->first();
        if ($transferTrans) {
            transactiontmp::where('transactiontmp_id', $transferTrans->transactiontmp_id)->delete();
            transactiondttmp::where('transactiondttmp_id', $transferTrans->transactiontmp_id)->delete();
        }

        return ApiController::successResponse($cancelTransactionTmp, 200);
    }

    public function removeTransactionDtSerial(Request $request)
    {
        $deleteSerial = transactionDtSnTmp::where('transactiondtsntmp_id', $request->transactiondtsntmp_id)->update(
            [
                "status" => 'Delete',

            ]
        );
        return ApiController::successResponse($deleteSerial, 200);
    }

    public function transactionReport(Request $request)
    {
        $transactions = DB::table('transactions')
            ->whereRaw('date(FROM_UNIXTIME(transaction_date)) BETWEEN ? AND ?', [$request->fromDate, $request->toDate])
            ->where([
                ['transaction_type', '=', $request->transactionType]
            ]);
        if ($request->transactionClientId) {

            $transactions->where([
                ['transaction_clientId', '=', $request->transactionClientId]
            ]);
        }
        if ($request->transactionCustomerId) {

            $transactions->where([
                ['clientCustomer_id', '=', $request->transactionCustomerId]
            ]);
        }
        if ($request->forwarderId) {

            $transactions->where([
                ['forwarder_id', '=', $request->forwarderId]
            ]);
        }
        $transactionData = $transactions->join('clients', 'clients.client_id', '=', 'transactions.transaction_clientId')
            ->leftjoin('client_customers', 'client_customers.customer_id', '=', 'transactions.clientCustomer_id')
            ->leftjoin('client_forwarders', 'client_forwarders.clientforwarder_id', '=', 'transactions.forwarder_id')
            ->get();
        if ($request->fecthDts) {
            foreach ($transactionData as $transaction) {
                $transactionDt = DB::table('transaction_dts')
                    ->where('transaction_id', '=', $transaction->transaction_id)
                    ->get();
                $transaction->transactionDt = $transactionDt;
            }
        }


        return ApiController::successResponse($transactionData, 200);
    }

    public function returnToTmpTransaction(Request $request)
    {

        $loggerUser = auth()->user()->id;
        $commonServices = new CommonServices();


        $realesed = transactions::where('transaction_id', '=', $request->transactionId)
            ->first();
        if ($realesed->transaction_released) {

            return ApiController::errorResponse("Already Released", 422);
        }


        $checkExsistance = transactiontmp::where('transactiontmp_id', '=', $request->transactionId)
            ->join('users', 'users.id', '=', 'transactiontmps.user_id')
            ->join('clients', 'clients.client_id', '=', 'transactiontmps.transactiontmp_clientId')
            ->leftjoin('client_customers', 'client_customers.customer_id', '=', 'transactiontmps.clientCustomer_id')
            ->leftjoin('client_forwarders', 'client_forwarders.clientforwarder_id', '=', 'transactiontmps.forwarder_id')
            ->orderBy('transactiontmp_date', 'asc')
            ->first();


        if ($checkExsistance) {
            $type = "Goods In";
            if ($checkExsistance->transactiontmp_type == "DO") {
                $type = "Delivery Order";
            }

            return ApiController::errorResponse($type . ' already exists in drafts !(By : ' . $checkExsistance->user_name . ')', 422);
        }


        if (!$checkExsistance) {
            $transactionId = $request->transactionId;
            $originalTransaction = $commonServices->returnHeaderOriginalToTmpService($transactionId);


            $originalTransactiondt = transactiondt::where('transaction_id', '=', $request->transactionId)->get();

            //$newTransactiondtTmp = transactiondttmp::create($originalTransactiondt);

            foreach ($originalTransactiondt as $data) {
                if ($originalTransaction->transaction_id && $data->transactiondt_id) {
                    $transactiondtTmp = array(
                        'transactiontmp_id' => $originalTransaction->transaction_id,
                        // 'transactiondttmp_price' => $originalTransaction->transactiondt_price,

                        'item_id' => $data->item_id,
                        'client_id' => $data->client_id,
                        'item_quantity' => $data->item_quantity,
                        'item_price' => $data->item_price,
                        'transactiondttmp_id' => $data->transactiondt_id,
                        'transactiontmpdt_billOfEntry' => $data->transactiondt_billOfEntry,
                        'transactiondttmp_countryOfOrigin' => $data->transactiondt_countryOfOrigin,
                        'transactiondttmp_price' => $data->transactiondt_price,
                        'transactiondttmp_priceBase' => $data->transactiondt_priceBase,
                        'id_reference_dt' => $data->id_reference_dt,
                        'raw_in_grid' => $data->raw_in_grid,

                        'supplier_id' => $data->supplier_id,
                        'supplier_invoice_number' => $data->supplier_invoice_number,
                        'transactiondttmp_locationid' => $data->transactiondt_locationid,
                        'item_weight' => $data->item_weight,
                        'item_unit' => $data->item_unit,

                        'item_qtyPerPack' => $data->item_qtyPerPack,
                        'item_pack' => $data->item_pack,
                        'group_id' => $data->group_id,
                        'palet_no' => $data->palet_no,
                        'transactiondttmp_nbOfCarton' => $data->transactiondt_nbOfCarton,
                        'transactiondttmp_cellMerge' => $data->transactiondt_cellMerge,
                        'transactiontmpdt_BOE' => $data->transactiondt_BOE,

                        'transactiondttmp_fprice' => $data->transactiondt_fprice,
                        'transactiondttmp_fpriceBase' => $data->transactiondt_fpriceBase,
                        'hscode' => $data->hscode,
                        'transactiondt_item_description' => $data->transactiondt_item_description,
                        'total_weight' => $data->total_weight,
                        'total_price' => $data->total_price,
                        'item_quantity_to_delivery' => $data->item_quantity_to_delivery,
                        'item_quantity_to_cancel' => $data->item_quantity_to_cancel,
                        'dt_transaction_type' => $data->dt_transaction_type,
                        'reference_from_do' => $data->reference_from_do,
                        'reference_internal_dt' => $data->reference_internal_dt,
                        'total_gross_weight' => $data->total_gross_weight,
                        'gross_weight' => $data->gross_weight,


                    );
                    $newTransactiondtTmp = transactiondttmp::create($transactiondtTmp);


                    $serials = transactionDtsn::where('transactiondt_id', '=', $newTransactiondtTmp->transactiondttmp_id)->get();

                    foreach ($serials as $data) {
                        $transactiondtSnTmp = array(
                            'transactiondttmp_id' => $data->transactiondt_id,
                            'transactiondtsntmp_id' => $data->transactiondtsn_id,
                            'item_id' => $data->item_id,
                            'transaction_item_serial' => $data->transaction_item_serial,

                        );
                        $newTransactionSnTmp = transactionDtSnTmp::create($transactiondtSnTmp);
                    }
                }
            }
            $palets = DB::table('transaction_palets')
                ->where([
                    ['transaction_id', $originalTransaction->transaction_id]
                ])
                ->get();
            foreach ($palets as $paletsTmp) {

                $palets = array(
                    'palet_no' => $paletsTmp->palet_no,
                    'palet_dimension' => $paletsTmp->palet_dimension,
                    'transactiontmp_id' => $paletsTmp->transaction_id,
                    'palet_weight' => $paletsTmp->palet_weight,
                    'palet_order' => $paletsTmp->palet_order,

                );
                TransactiontmpPalets::create($palets);
            }
            // if ($newTransactionTmp->transactiontmp_date) {
            //     $newTransactionTmp->transactiontmp_formatdate = date('Y-m-d', $newTransactionTmp->transactiontmp_date);
            // }

            $storageData = DB::table('client_storage_pallets')
                ->where('related_transaction_id', $request->transactionId)
                ->get();
            foreach ($storageData as $item) {
                $data = array(
                    'length' => $item->length,
                    'width' => $item->width,
                    'height' => $item->height,
                    'related_transaction_id' => $item->related_transaction_id,
                    'pallet_no' => $item->pallet_no,
                    'pallet_seq' => $item->pallet_seq,
                    'related_plan' => $item->related_plan,
                    'id' => $item->id,

                );
                DB::table('client_storage_pallets_tmp')
                    ->insert($data);
            }

            $updateTransactionTmp = transactiontmp::where('transactiontmp_id', $request->transactionId)
                ->update(['user_id' => $loggerUser]);


            ///tmp to resolve
            $checkExsistance = transactiontmp::where('transactiontmp_id', '=', $request->transactionId)
                ->join('users', 'users.id', '=', 'transactiontmps.user_id')
                ->join('clients', 'clients.client_id', '=', 'transactiontmps.transactiontmp_clientId')
                ->leftjoin('client_customers', 'client_customers.customer_id', '=', 'transactiontmps.clientCustomer_id')
                ->leftjoin('client_forwarders', 'client_forwarders.clientforwarder_id', '=', 'transactiontmps.forwarder_id')
                ->orderBy('transactiontmp_date', 'asc')
                ->first();

            if ($checkExsistance->transactiontmp_date) {
                $checkExsistance->transactiontmp_formatdate = date('Y-m-d', $checkExsistance->transactiontmp_date);
            }

            $updateTransactionTmp = transactiontmp::where('transactiontmp_id', $request->transactionId)
                ->update(['user_id' => $loggerUser]);
            return ApiController::successResponse($checkExsistance, 200);
        }


        if ($checkExsistance->transactiontmp_date) {
            $checkExsistance->transactiontmp_formatdate = date('Y-m-d', $checkExsistance->transactiontmp_date);
        }


        $updateTransactionTmp = transactiontmp::where('transactiontmp_id', $request->transactionId)
            ->update(['user_id' => $loggerUser]);

        return ApiController::successResponse($checkExsistance, 200);
    }

    public function removeTransactionDtTmpDOFromMObile(Request $request)
    {

        $transactionDtTmps = DB::table('transactiondttmps')
            ->where([
                ['transactiontmp_id', '=', $request->transactionTmpId],
                ['item_id', '=', $request->itemId],
                ['isMobile', true],
            ])
            ->first();


        $transactionDtTmpsRemouved = DB::table('transactiondttmps')
            ->where([
                ['transactiontmp_id', '=', $request->transactionTmpId],
                ['item_id', '=', $request->itemId],
                ['isMobile', true],
            ])
            ->delete();

        return ApiController::successResponse($transactionDtTmpsRemouved, 200);
    }

    public function checkTransactionUpdateValidity(Request $request)
    {

        $transactionId = $request->transactionId;
        $billOfEntry = $request->billOfEntry;


        $checkDO = transactionDt::where([
            ['transactiondt_billOfEntry', $billOfEntry],
            ['transaction_id', '!=', $transactionId]
        ])
            ->leftjoin('items', 'items.item_id', 'transaction_dts.item_id')
            ->select('transaction_dts.*', 'items.item_code')
            ->get();


        if (count($checkDO) > 0) {
            return ApiController::errorResponse($checkDO, 422);
        }
        return ApiController::successResponse(true, 200);
    }

    public function addDOFromGoodsIn(Request $request)
    {
        $utilitiesController = new UtilitiesController();
        $transactionId = $request->transactionId;
        $loggerUserCompany = auth()->user()->company_id;
        $exitPoint = companies::where('company_id', $loggerUserCompany)
            ->first();
        $Transaction = transactions::where(
            [
                ['transaction_id', $transactionId],

            ]
        )
            ->first();


        $date = $utilitiesController->convertTodayToTimetsamp(date('Y-m-d H:i:s'));


        $transactionTmp = array(
            'transactiontmp_date' => $date,
            'transactiontmp_billOfEntry' => $Transaction->transaction_billOfEntry,

            'transactiontmp_clientId' => $Transaction->transaction_clientId,
            'transactiontmp_type' => $Transaction->transaction_type == 'GOODS_IN' ? 'DO' : 'RMA_OUT',
            'transactiontmp_remark' => 'Automatic creation from GoodsIn',
            'user_id' => $Transaction->user_id,
            'company_id' => $Transaction->company_id,
            'transactiontmp_currencyId' => $Transaction->transaction_currencyId,
            'transactiontmp_rate' => $Transaction->transaction_rate,
            'transactiontmp_exitPoint' => $exitPoint->company_exitCode,
        );
        $newTransactionTmp = transactiontmp::create($transactionTmp);


        $transactionsDtsOrignal = transactionDt::where(
            [
                ['transaction_id', $transactionId]
            ]
        )
            //->groupBy('item_id','transactiondt_billOfEntry','transactiondt_countryOfOrigin')
            ->get();
        $k = 0;
        $transactionSerialArray = [];
        foreach ($transactionsDtsOrignal as $Transactiondt) {
            if (!empty($Transactiondt)) {
                $itemDetails = items::where(
                    [
                        ['item_id', $Transactiondt->item_id]
                    ]
                )->first();

                $totalEntry = transactionDt::where(
                    [
                        ['transactiondt_id', $Transactiondt->transactiondt_id]
                    ]
                )
                    ->sum('item_quantity');

                $totalExit = transactionDt::where(
                    [
                        ['id_reference_dt', $Transactiondt->transactiondt_id],
                        ['dt_transaction_type', '!=', 'DQ'],

                    ]
                )->sum('item_quantity');
                //  $total=$totalEntry-$totalExit;
                $totalEntry = isset($totalEntry) ? $totalEntry : 0;
                $totalExit = isset($totalExit) ? $totalExit : 0;
                $total = $totalEntry - $totalExit;


                // Formulate record that will be saved
                $transactionDtsTmp = [];
                $transactionDtsTmp = array(
                    'transactiontmp_id' => $newTransactionTmp->transactiontmp_id,
                    'item_id' => $Transactiondt->item_id,
                    'item_quantity' => $total,
                    'item_price' => $Transactiondt->item_price,
                    'group_id' => $itemDetails->group_id,
                    'item_weight' => $Transactiondt->item_weight,
                    'item_pack' => $itemDetails->item_pack,
                    'item_unit' => $itemDetails->item_unit,
                    'item_qtyPerPack' => $itemDetails->item_qtyPerPack,
                    'id_reference_dt' => $Transactiondt->transactiondt_id,
                    'raw_in_grid' => $k,
                    'transactiondttmp_cellMerge' => $k + 1,
                    'transactiondttmp_nbOfCarton' => (isset($itemDetails->item_qtyPerPack) && $itemDetails->item_qtyPerPack > 0) ? ceil($Transactiondt->item_quantity / $itemDetails->item_qtyPerPack) : 0,
                    'supplier_id' => $Transactiondt->supplier_id,
                    'supplier_invoice_number' => $Transactiondt->supplier_invoice_number,
                    'transactiondttmp_locationid' => $Transactiondt->transactiondt_locationid,
                    'transactiondttmp_countryOfOrigin' => $Transactiondt->transactiondt_countryOfOrigin,
                    'transactiontmpdt_billOfEntry' => $Transactiondt->transactiondt_billOfEntry,
                    'transactiondttmp_price' => $Transactiondt->transactiondt_price,
                    'client_id' => $Transactiondt->client_id,
                    'hscode' => $Transactiondt->hscode,
                    'transactiondt_item_description' => $Transactiondt->transactiondt_item_description,
                    'transactiondttmp_fpriceBase' => $Transactiondt->transactiondt_fpriceBase,
                    'transactiondttmp_fprice' => $Transactiondt->transactiondt_fprice,
                    'transactiondttmp_priceBase' => $Transactiondt->transactiondt_priceBase,
                    'transactiontmpdt_BOE' => $Transactiondt->transactiondt_BOE,
                    'dt_transaction_type' => 'DO',
                    'total_gross_weight' => $Transactiondt->total_gross_weight,
                    'gross_weight' => $Transactiondt->gross_weight,


                );
            }
            $k = $k + 1;
            $newTransactionDtsTmp = transactiondttmp::create($transactionDtsTmp);

            $serials = transactionDtSn::where([
                ['transactiondt_id', $Transactiondt->transactiondt_id]
            ])->get();
            foreach ($serials as $splitSerials) {
                $transactionSerialArray[] = [
                    'item_id' => $splitSerials->item_id,
                    'transaction_item_serial' => $splitSerials->transaction_item_serial,
                    'transactiondttmp_id' => $newTransactionDtsTmp->transactiondttmp_id,
                ];
            }
        }

        $newTransactionSerialsTmp = transactionDtSnTmp::insert($transactionSerialArray);

        $type = $Transaction->transaction_type == 'GOODS_IN' ? 'DO' : 'RMA OUT';
        $message = ' Automatic ' . $type . ' was created';

        return ApiController::successResponse($message, 200);
    }

    public function uploadDocumentsByTransaction(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'transactionId' => 'required',
            'transactionType' => 'required'
        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 422);
        }

        $path = public_path() . '/documents/' . $request->transactionType . '/' . $request->transactionId . '/';
        if (!file_exists($path)) {
            File::makeDirectory($path, $mode = 0777, true, true);
        }
        $file = $request->file;
        $filename = $file->getClientOriginalName();
        $extension = $file->getClientOriginalExtension();
        $microTime = microtime(true);
        $milliseconds = intval($microTime * 1000);
        $document = 'FILE_' . $milliseconds . '.' . $extension;
        $insertStatus = $file->move($path, $document);

        return ApiController::successResponse($insertStatus, 200);
    }

    function fetchDocumentsByTransactionIdAndType(Request $request)
    {
        $path = public_path('/documents/' . $request->transactionType . '/' . $request->transactionId);
        $filesArray = array();
        if (file_exists($path)) {

            $files = array_slice(scandir($path), 2);

            for ($i = 0; $i < count($files); $i++) {
                $fileObject = array();
                $fileObject['file_name'] = $files[$i];
                $pathWithName = $path . '/' . $files[$i];
                $pathWithNameReturned = $files[$i];
                $pathWithName = str_replace(config('app.IMAGE_PATH'), config('app.IMAGE_PATH_REPLACE'), $pathWithName);
                $fileObject['path'] = $pathWithName;
                $fileObject['pathWithNameReturned'] = 'documents/' . $request->transactionType . '/' . $request->transactionId . '/' . $pathWithNameReturned;
                if (
                    str_contains($fileObject['file_name'], '.xlsx') || str_contains($fileObject['file_name'], '.docx')
                    || str_contains($fileObject['file_name'], '.pdf')
                ) {
                    $fakePath = 'office';
                    if (str_contains($fileObject['file_name'], '.xlsx')) {
                        $imageName = $fakePath . '/excel.png';
                    } else if (str_contains($fileObject['file_name'], '.docx')) {
                        $imageName = $fakePath . '/word.png';
                    } else {
                        $imageName = $fakePath . '/pdf.png';
                    }

                    $fileObject['fake_path'] = $imageName;
                } else {
                    $fileObject['fake_path'] = null;
                }

                array_push($filesArray, $fileObject);
            }
        }
        return ApiController::successResponse($filesArray, 200);
    }

    public function removeDocument(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'file' => 'required',
            'transactionId' => 'required',
            'transactionType' => 'required'
        ]);
        if ($validator->fails()) {
            return response()->json($validator->errors(), 422);
        }

        $path = public_path() . '/documents/' . $request->transactionType . '/' . $request->transactionId . '/' . $request->file;

        if (file_exists($path)) {
            $deleteFile = File::delete($path);
            return ApiController::successResponse($deleteFile, 200);
        }
        return ApiController::errorResponse($validator->errors(), 422);
    }

    public function storeTransactionDtTmpGoodsIn(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'transactionDetailsTmp' => 'required|array',
        ]);
        if ($validator->fails()) {
            return response()->json($validator->errors(), 422);
        }


        $transactionDetailsTmp = $request->transactionDetailsTmp;
        $transactionTmpId = $request->transactionTmpId;
        return $this->storeTransactionDtTmpFunction($transactionDetailsTmp, $transactionTmpId);
    }

    public function storeTransactionDtTmpFunction($transactionDetailsTmp, $transactionTmpId)
    {

        $newTransactionDtTmp = null;
        $goodsInBillOfEntry = null;
        $goodsInFunctions = new GoodsInFunctions();


        foreach ($transactionDetailsTmp as $element) {

            if ($element['transactionDetailItemCode'] != null && $element['transactionDetailItemQty'] != null) {
                $getBillOfEntryInCaseOfGoodsIn = transactiontmp::where('transactiontmp_id', '=', $transactionTmpId)->first();

                if ($getBillOfEntryInCaseOfGoodsIn) {
                    $goodsInBillOfEntry = $getBillOfEntryInCaseOfGoodsIn->transactiontmp_billOfEntry;
                }

                $getItem = Items::where('item_code', trim($element['transactionDetailItemCode']))
                    ->first();
                $item = null;
                $invalidData = array();
                if ($getItem) {
                    $item = $getItem->item_id;
                } else {
                    array_push($invalidData, $element['transactionDetailItemCode'] . "not saved");
                    continue;
                }

                $transactionDtTmp = $goodsInFunctions->prepareTransactionDtTmp($getBillOfEntryInCaseOfGoodsIn, $element, $goodsInBillOfEntry, $transactionTmpId, $item);

                if (!isset($element['transactiondttmp_id']) || $element['transactiondttmp_id'] == null) {
                    $newTransactionDtTmp = transactiondttmp::create($transactionDtTmp);
                } else {
                    $updateTransactionDtTmp = transactiondttmp::where('transactiondttmp_id', '=', $element['transactiondttmp_id'])
                        ->update($transactionDtTmp);

                    transactionDtSnTmp::where('transactiondttmp_id', '=', $element['transactiondttmp_id'])
                        ->update([
                            'item_id' => $item
                        ]);


                    $newTransactionDtTmp = transactiondttmp::where('transactiondttmp_id', '=', $element['transactiondttmp_id'])
                        ->leftjoin('items', 'items.item_id', 'transactiondttmps.item_id')
                        ->select('transactiondttmps.*', 'items.item_hasSerial')
                        ->first();
                    if (!$newTransactionDtTmp->item_hasSerial) {
                        transactionDtSnTmp::where('transactiondttmp_id', '=', $element['transactiondttmp_id'])
                            ->update([
                                'status' => 'Delete'
                            ]);
                    }
                }

                $transactionDtSumQty = transactiondttmp::where('transactiondttmps.transactiontmp_id', '=', $transactionTmpId)
                    ->select('transactiondttmps.item_quantity')
                    ->selectRaw('SUM(transactiondttmps.item_quantity) as totalQty')
                    ->first();

                transactiontmp::where('transactiontmps.transactiontmp_id', '=', $transactionTmpId)
                    ->update(["transactiontmp_totalQty" => $transactionDtSumQty->totalQty]);
            }
        }

        if ($newTransactionDtTmp) {
            $data = array();
            $data['transactionDt'] = $newTransactionDtTmp;
            $data['transactionTmp'] = $newTransactionDtTmp;
            $data['invalidData'] = $invalidData;

            return ApiController::successResponse($data, 200);
        }

        return ApiController::errorResponse($newTransactionDtTmp, 422);
    }


    public function ExportDraftGoodIn(Request $request)
    {
        $company_id = JWTAuth::user()->company_id;
        $commonServices = new CommonServices();
        $dateFormat = $commonServices->dubaiTradeDAteFormatService();

        $transactionId = $request->transactionId;
        $invoiceId = $request->invoiceId;
        $paymentTermId = $request->paymentTermId;
        $deliveryTermId = $request->deliveryTermId;
        $exemptionId = $request->exemptionId;
        $trade = "";
        transactiontmp::where([
            ['transactiontmp_id', $transactionId]
        ])
            ->update([
                'Invoice_type_id' => $invoiceId,
                'payment_term_id' => $paymentTermId,
                'delivery_term_id' => $deliveryTermId,
                'exemption_type_id' => $exemptionId
            ]);
        $header = transactiontmp::where([
            ['transactiontmp_id', $transactionId]
        ])
            ->leftjoin('currency', 'currency_id', 'transactiontmps.transactiontmp_currencyId')
            ->leftjoin('client_suppliers', 'client_suppliers.supplier_id', 'transactiontmps.transactiontmp_supplierId')
            ->select(
                'delivery_term_id',
                'currency_code',
                'payment_term_id',
                'Invoice_type_id',
                'transactiontmp_id',
                'transactiontmps.transactiontmp_reference as transaction_reference',
                'transactiontmps.transactiontmp_type as transaction_type',
                'transactiontmps.transactiontmp_supplierInvoice as transaction_supplierInvoice',
                'client_suppliers.supplier_name'
            )
            ->first();
        $header->transaction_formatdate = date($dateFormat, $header->transactiontmp_date);
        $details = transactiondttmp::where([
            ['transactiontmp_id', $transactionId],
            ['state', '!=', 'deleted']
        ])
            ->leftjoin('items', 'items.item_id', 'transactiondttmps.item_id')
            ->leftjoin('countries', 'countries.country_id', 'transactiondttmps.transactiondttmp_countryOfOrigin')
            ->select('transactiondttmp_price as transactiondt_price', 'transactiondttmps.*', 'items.item_id', 'items.item_description', 'countries.country_code', 'items.item_unit')
            ->get();
        $header->totalValues = transactiondttmp::where([
            ['transactiontmp_id', $header->transactiontmp_id]
        ])
            ->selectRaw('SUM(transactiondttmp_price*item_quantity) as totalValues')
            ->first()->totalValues;

        $trade = $commonServices->dubaiTradeRecordService($header, null, null, true);

        $index = 1;
        foreach ($details as $splitDetails) {
            $trade = $trade . "\n" . $commonServices->dubaiTradeRecordService($header, $splitDetails, $index, false);
            $index = $index + 1;
        }

        return ApiController::successResponse($trade, 200);
    }


    public function importGoodIn(Request $request)
    {

        $file = $request->file('file');
        // Process the file (e.g., store it, read its contents, etc.)
        $contents = file_get_contents($file->path());
        $splitFile = explode("\n", $contents);
        $user_id = JWTAuth::user()->id;
        $companyId = auth()->user()->company_id;
        $clientId = $request->clientId;
        $utilitiesController = new UtilitiesController();
        $commonServices = new CommonServices();
        $dateFormat = $commonServices->dubaiTradeDAteFormatService();

        $dateFormat = str_replace('%', '', $dateFormat);

        $i = 0;
        $transactiontmp_id = 0;
        $header_response = null;
        $currencyCode = '';
        $currencyIdValue = 0;
        $currencyRateValue = 1;
        $errorArray = array();
        $errorMessage = '';
        if (count($splitFile) > 0) {

            foreach ($splitFile as $data) {
                if ($i == 0) {

                    $headerData = explode(",", $data);
                    $headerData = str_replace("\"", '', $headerData);
                    $date = $headerData[$commonServices->indexOfFieldsDubaiTraderService('headerDate')];
                    $currencyCode = $headerData[$commonServices->indexOfFieldsDubaiTraderService('headerCurrencyCode')];
                    $supplierInvoice = $headerData[$commonServices->indexOfFieldsDubaiTraderService('headerSupplier')];
                    $date = str_replace("\"", '', $date);
                    $date = Carbon::createFromFormat($dateFormat, $date);
                    $date = str_replace("\"", '', $date);
                    $dateConvert = $utilitiesController->convertTodayToTimetsamp($date);
                    $date = str_replace("\"", '', $date);
                    $currencyId = DB::table('currency')
                        ->where('currency_code', $currencyCode)
                        ->first();
                    if ($currencyId) {
                        $currencyIdValue = $currencyId->currency_id;
                        $currencyRate = $utilitiesController->getCurrencyRateFunction($currencyId->currency_id, $date);
                        $currencyRateValue = $currencyRate[0]->CurRate;
                    }

                    $currentDateTime = Carbon::now();
                    $formattedTime = $currentDateTime->format('H:i');
                    $transactionTmpArray = array(
                        'transactiontmp_date' => $dateConvert,
                        'transaction_time' => $formattedTime,
                        'transactiontmp_type' => 'GOODS_IN',
                        'transactiontmp_remark' => 'test import',
                        'transactiontmp_billOfEntry' => null,
                        'user_id' => $user_id,
                        'forwarder_id' => $request->orderForwarder,
                        'clientCustomer_id' => $request->transactionCustomer,
                        'company_id' => $companyId,
                        'transactiontmp_currencyId' => $currencyIdValue,
                        'transactiontmp_rate' => $currencyRateValue,
                        'transactiontmp_shipmentRemark' => $request->transactionShipmentRemark,
                        'transactiontmp_supplierId' => $request->transactionSupplier,
                        'transactiontmp_supplierInvoice' => $supplierInvoice,
                        'transactiontmp_BOE' => $request->transactionBOE != '' ? $request->transactionBOE : '-',
                        'transactiontmp_clientId' => $clientId
                    );
                    $header_response = transactiontmp::create($transactionTmpArray);
                    $header_response['client_id'] = (float)$clientId;
                    $header_response['transactiontmp_formatdate'] = date('Y-m-d', $header_response->transactiontmp_date);

                    $transactiontmp_id = $header_response->transactiontmp_id;
                    $i++;
                } else {
                    $details = explode(",", $data);
                    $details = str_replace("\"", '', $details);

                    //      '"ID"' . ',"' . $index . '","' . '' . $splitDetails->hscode . '"' . "," . '"' . $splitDetails->item_description . '"' . "," . '"N",' . '"' . $splitDetails->item_unit . '","'.$splitDetails->item_quantity.'",' . '"kg",' . '"' . $grossWeight . '"' . "," . '"","",' . '"' . $grossPrice . '",' . '"' . $splitDetails->country_code . '",' . '"",' . '"","",' . '"",' . '"' . $header->exemption_type_id . '",' . '""';
                    if (count($details) < 2) continue;

                    $itemCode = $details[$commonServices->indexOfFieldsDubaiTraderService('itemCodeDetail')];
                    $description = $details[$commonServices->indexOfFieldsDubaiTraderService('descriptionDetail')];
                    $qty = $details[$commonServices->indexOfFieldsDubaiTraderService('qtyDetail')];
                    $grossWeight = $details[$commonServices->indexOfFieldsDubaiTraderService('grossWeight')];
                    $grossPrice = $details[$commonServices->indexOfFieldsDubaiTraderService('grossPrice')];
                    $countryCode = $details[$commonServices->indexOfFieldsDubaiTraderService('countryCode')];
                    $price = (float)$grossPrice / (float)$qty;
                    $weight = (float)$grossWeight / (float)$qty;
                    $item = Items::where('item_hscode', $itemCode)
                        ->first();
                    if (!$item) {
                        $errorMessage = $errorMessage . "\n" . 'Item ' . $itemCode . ' not found';
                        array_push($errorArray, $itemCode);
                        continue;
                    }
                    $country = DB::table('countries')->where('country_code', $countryCode)
                        ->first();
                    $itemId = 0;
                    $countryId = 0;
                    if ($item) {
                        $itemId = $item->item_id;
                    } else {
                    }
                    if ($country) {
                        $countryId = $country->country_id;
                    }
                    $transactiondtTmp = array(
                        'transactiontmp_id' => $transactiontmp_id,
                        'transactiontmpdt_BOE' => null,
                        'item_id' => $itemId,
                        'group_id' => null,
                        'transactiondt_item_description' => $description,
                        'item_quantity' => $qty,
                        'transactiontmpdt_billOfEntry' => null,
                        'transactiondttmp_locationid' => null,
                        'transactiondttmp_countryOfOrigin' => $countryId,
                        'transactiondttmp_price' => $price,
                        'transactiondttmpPriceBase' => $price * $currencyRateValue,
                        'id_reference_dt' => null,
                        'raw_in_grid' => null,
                        'palet_no' => null,
                        'item_weight' => $weight,
                        'gross_weight' => $weight,
                        'item_qtyPerPack' => null,
                        'item_pack' => " ",
                        'transactiondttmp_nbOfCarton' => " ",
                        'transactiondttmp_cellMerge' => " ",
                        'item_unit' => null,
                        'client_id' => $clientId,
                        'transactiondttmp_fprice' => 0,
                        'transactiondttmp_fpriceBase' => 0,
                        'hscode' => $itemCode,
                        'supplier_invoice_number' => null,
                        'total_price' => $grossPrice,
                        'total_weight' => $grossWeight,
                        'total_gross_weight' => $grossWeight,
                        'state' => 'new',
                        'supplier_id' => null,

                    );
                    $detail_response = transactiondttmp::create($transactiondtTmp);
                }
            }
        }
        $obj = array();
        $obj['data'] = $header_response;
        $obj['error'] = $errorArray;
        $obj['errorMessage'] = $errorMessage;

        return ApiController::successResponse($obj, 200);
    }


    public function fetchRemainingGoodsInTransactions(Request $request)
    {
        $utilitiesController = new UtilitiesController();
        $transactionId = $request->transactionId;
        $fromClient = $request->fromClient;
        $search = $request->search;

        $date = $request->toDate;

        $transactions = transactions::leftJoin('clients', 'clients.client_id', '=', 'transactions.transaction_clientId')
            ->leftJoin('suppliers', 'suppliers.supplier_id', '=', 'transactions.transaction_supplierId')
            ->leftJoin('client_customers', 'client_customers.customer_id', '=', 'transactions.clientCustomer_id')
            ->select(
                'customer_name',
                'transactions.transaction_reference',
                'transactions.transaction_date',
                'clients.client_firstname',
                'transactions.transaction_id',
                'transactions.transaction_mobileReference',
                'salesjv_id',
                'transactions.transaction_remark',
                'transactions.transaction_billOfEntry',
                'transactions.transaction_shipmentReference',
                'transactions.transaction_typeOfDeliveryId',
                'suppliers.supplier_name',
                'transactions.transaction_supplierInvoice',
                'transactions.transaction_type',
                'transaction_BOE',
                'Invoice_type_id',
                'payment_term_id',
                'delivery_term_id',
                'exemption_type_id',
                'internal_transfer',
            )
            ->leftjoin('transaction_dts as g', 'g.transaction_id', 'transactions.transaction_id')
            ->leftjoin('transaction_dts as out', 'out.id_reference_dt', 'g.transactiondt_id')
            ->leftjoin('transactiondttmps as outTmp', 'outTmp.id_reference_dt', 'g.transactiondt_id')
            ->selectRaw('SUM( COALESCE(g.item_quantity, 0)) as totalQty')
            ->distinct('g.id')
            ->selectRaw('SUM(COALESCE(out.item_quantity, 0)) as totalQtyExit')
            ->selectRaw('SUM(CASE WHEN outTmp.transactiontmp_id = ' . $transactionId . ' THEN COALESCE(outTmp.item_quantity, 0) ELSE 0 END) as totalQtyExitTmp')
            ->selectRaw("FROM_UNIXTIME(transactions.transaction_date, '%d-%m-%Y') AS transaction_formatdate")
            ->selectRaw("
            CASE
                WHEN transactions.transaction_type = 'GOODS_IN' AND transactions.internal_transfer = 1 THEN 'Internal'
                ELSE ''
            END as transactionType
        ")
            ->where([
                ['transactions.transaction_type', 'GOODS_IN'],
                ['transactions.transaction_id', '<>', $transactionId],

            ])
            ->whereRaw('DATE(FROM_UNIXTIME(transactions.transaction_date)) <= DATE(FROM_UNIXTIME(' . $date . '))')
            ->when($fromClient, fn($query) => $query->where('transaction_clientId', $fromClient))
            ->when($search, fn($query) => $query->where('transaction_reference', 'like', '%' . $search . '%'))
            ->havingRaw('totalQty > (totalQtyExit+totalQtyExitTmp)')
            ->groupBy('transactions.transaction_id')
            ->orderBy('transactions.transaction_date', 'desc')
            ->orderBy('transactions.transaction_reference', 'desc');

        $transactions = $transactions->get();

        return ApiController::successResponse($transactions, 200);
    }


    public function addDOFromMultipleGoodsIn(Request $request)
    {
        $transactions = $request->transactions;
        $transactionTmpId = $request->transactionTmpId;
        $commonServices = new CommonServices();
        $loadPrice = $commonServices->checkLoadedPriceModule();
        $k = 0;

        $maxIndex = transactiondttmp::where(
            [
                ['transactiontmp_id', $transactionTmpId]
            ]
        )
            ->max('raw_in_grid');

        if ($maxIndex !== null) {
            $k = $maxIndex + 1;
        }

        foreach ($transactions as $transElement) {
            $transactionId = $transElement['transaction_id'];
            $Transaction = transactions::where('transaction_id', $transactionId)
                ->first();
            $transactionsDtsOrignal = transactionDt::where('transaction_id', $transactionId)
                ->get();

            $transactionSerialArray = [];
            foreach ($transactionsDtsOrignal as $Transactiondt) {
                if (!empty($Transactiondt)) {
                    $itemDetails = items::where(
                        [
                            ['item_id', $Transactiondt->item_id]
                        ]
                    )->first();

                    $totalEntry = transactionDt::where(
                        [
                            ['transactiondt_id', $Transactiondt->transactiondt_id]
                        ]
                    )
                        ->sum('item_quantity');

                    $totalExit = transactionDt::where(
                        [
                            ['id_reference_dt', $Transactiondt->transactiondt_id],
                            ['transaction_id', '!=', $transactionTmpId]
                        ]
                    )->sum('item_quantity');

                    $totalExitTmp = transactiondttmp::where(
                        [
                            ['id_reference_dt', $Transactiondt->transactiondt_id],
                            ['transactiontmp_id', $transactionTmpId]
                        ]
                    )->sum('item_quantity');

                    $totalEntry = isset($totalEntry) ? $totalEntry : 0;
                    $totalExit = isset($totalExit) ? $totalExit : 0;
                    $totalExitTmp = isset($totalExitTmp) ? $totalExitTmp : 0;
                    $total = $totalEntry - $totalExit - $totalExitTmp;

                    // Formulate record that will be saved
                    $transactionDtsTmp = [];
                    if ($total <= 0) continue;

                    $transactionDtsTmp = array(
                        'transactiontmp_id' => $transactionTmpId,
                        'item_id' => $Transactiondt->item_id,
                        'item_quantity' => $total,
                        'item_price' => $Transactiondt->item_price,
                        'group_id' => $itemDetails->group_id,
                        'item_weight' => $Transactiondt->item_weight,
                        'item_pack' => $itemDetails->item_pack,
                        'item_unit' => $itemDetails->item_unit,
                        'item_qtyPerPack' => $itemDetails->item_qtyPerPack,
                        'id_reference_dt' => $Transactiondt->transactiondt_id,
                        'raw_in_grid' => $k,
                        'transactiondttmp_cellMerge' => $k + 1,
                        'transactiondttmp_nbOfCarton' => (isset($itemDetails->item_qtyPerPack) && $itemDetails->item_qtyPerPack > 0) ? ceil($total / $itemDetails->item_qtyPerPack) : 0,
                        'supplier_id' => $Transaction->transaction_supplierId,
                        'supplier_invoice_number' => $Transaction->transaction_supplierInvoice,
                        'transactiondttmp_locationid' => $Transactiondt->transactiondt_locationid,
                        'transactiondttmp_countryOfOrigin' => $Transactiondt->transactiondt_countryOfOrigin,
                        'transactiontmpdt_billOfEntry' => $Transactiondt->transactiondt_billOfEntry,
                        'transactiondttmp_price' => $loadPrice == 1 ? $Transactiondt->transactiondt_priceBase : 0,
                        'client_id' => $Transactiondt->client_id,
                        'hscode' => $Transactiondt->hscode,
                        'transactiondt_item_description' => $Transactiondt->transactiondt_item_description,
                        'transactiondttmp_fpriceBase' => $Transactiondt->transactiondt_fpriceBase,
                        'transactiondttmp_fprice' => $Transactiondt->transactiondt_fprice,
                        'transactiondttmp_priceBase' => $Transactiondt->transactiondt_priceBase,
                        'transactiontmpdt_BOE' => $Transactiondt->transactiondt_BOE,
                        'dt_transaction_type' => 'DO',
                        'gross_weight' =>  $Transactiondt->gross_weight,
                        'total_gross_weight' => $Transactiondt->total_gross_weight,
                        'total_weight' => $Transactiondt->total_weight,
                        'total_price' => $Transactiondt->total_price,

                    );
                }
                $k = $k + 1;
                $newTransactionDtsTmp = transactiondttmp::create($transactionDtsTmp);


                $serials = transactionDtSn::where([
                    ['transactiondt_id', $Transactiondt->transactiondt_id]
                ])->get();
                transactionDtSnTmp::where([
                    ['transactiondttmp_id', $newTransactionDtsTmp->transactiondttmp_id,]
                ])->delete();
                foreach ($serials as $splitSerials) {
                    $transactionSerialArray[] = [
                        'item_id' => $splitSerials->item_id,
                        'transaction_item_serial' => $splitSerials->transaction_item_serial,
                        'transactiondttmp_id' => $newTransactionDtsTmp->transactiondttmp_id,
                    ];
                }
            }

            transactionDtSnTmp::insert($transactionSerialArray);

            $type = $Transaction->transaction_type == 'GOODS_IN' ? 'DO' : 'RMA OUT';
            $message = ' Automatic ' . $type . ' was created';
        }

        $transactions = transactiontmp::join('users', 'users.id', '=', 'transactiontmps.user_id')
            ->join('clients', 'clients.client_id', '=', 'transactiontmps.transactiontmp_clientId')
            ->leftjoin('client_customers', 'client_customers.customer_id', '=', 'transactiontmps.clientCustomer_id')
            ->leftjoin('client_forwarders', 'client_forwarders.clientforwarder_id', '=', 'transactiontmps.forwarder_id')
            ->orderBy('transactiontmp_date', 'desc')
            ->where([
                ['transactiontmp_id', $transactionTmpId],
            ])
            ->first();

        $transactions->transactiontmp_formatdate = date('Y-m-d', $transactions->transactiontmp_date);

        return ApiController::successResponse($transactions, 200);
    }

    public function importTransactionDetailsFromExcel(Request $request)
    {
        $records = $request->data; 
        $rate = $request->rate;
    
        $utilitiesController = new UtilitiesController();
        $company = companies::first();
    
        $validRecords = [];
        $invalidData = [];
        $transactionTmpId = $request->transactionId;
        $transactionTmp = transactiontmp::where('transactiontmp_id', '=', $transactionTmpId)->first();
    
        $headers = $records[0];
        unset($records[0]); 
            $headerMap = array_flip($headers);
    
        foreach ($records as $index => $record) {
            if (empty(array_filter($record))) {
                continue; 
            }
    
            $invalidRecord = ['Record' => $index + 1, 'message' => ''];
            $itemCode = $record[$headerMap['item_code']] ?? null;
            $itemDescription = $record[$headerMap['item_description']] ?? null;
            $hscode = $record[$headerMap['hscode']] ?? null;
            $countryCode = $record[$headerMap['country Of Origin code']] ?? null;
            $quantity = $record[$headerMap['QTY']] ?? null;
            $price = $record[$headerMap['PRICE']] ?? null;
            $totalWeight = $record[$headerMap['WEIGHT']] ?? null;
            $itemPack = $record[$headerMap['PACK']] ?? '-';
            $qtyPerPack = $record[$headerMap['QTY/PACK']] ?? 1;
            $itemUnit =isset($headerMap['ITEM_UNIT']) ?  $record[$headerMap['ITEM_UNIT']]:'u';
            $totalWeight = $record[$headerMap['WEIGHT']] ?? null;
            $grossWeight = isset($headerMap['Gross_Weight']) && $record[$headerMap['Gross_Weight']] ? $record[$headerMap['Gross_Weight']]:null ;
    
            $item = items::where('item_code', trim($itemCode))->first();
            $country = country::where('country_code', trim($countryCode))
                ->orWhere('country_name', trim($countryCode))
                ->first();
    
            $errorMessages = [];
             $errorMessages[] = $utilitiesController->validateItem($item);
             $errorMessages[] = $utilitiesController->validateQuantity($quantity);
             $errorMessages[] = $utilitiesController->validatePrice($price);
             $errorMessages[] = $utilitiesController->validateWeight($totalWeight);
             $errorMessages[] = $utilitiesController->validateCountry($country);
    
            if ($company->company_grossWeight === 1) {
                $errorMessages[] = $utilitiesController->validateGrossWeight($totalWeight,$grossWeight);
            }else{
                $grossWeight= $totalWeight;
            }
    
            $errorMessages = array_filter($errorMessages);
    
            if (!empty($errorMessages)) {
                $invalidRecord['Item'] = $itemCode;
                $invalidRecord['message'] = implode(', ', $errorMessages);
                $invalidData[] = $invalidRecord;
                continue;
            }
    
            // Prepare transaction details
            $transactionDtArray = [
                'transactiontmp_id' => $transactionTmpId,
                'item_id' => $item->item_id,
                'transactiondt_item_description' => $itemDescription,
                'hscode' => $hscode,
                'item_quantity' => $quantity,
                'transactiondttmp_countryOfOrigin' => $country->country_id ?? null,
                'transactiondttmp_price' => $price / $quantity,
                'transactiondttmp_priceBase' => ($price / $quantity) * $rate,
                'item_weight' => $totalWeight / $quantity,
                'total_weight' => $totalWeight,
                'gross_weight' => $grossWeight / $quantity,
                'total_gross_weight' => $grossWeight,
                'total_price' => $price,
                'client_id' => $transactionTmp->transactiontmp_clientId,
                'transactiontmpdt_BOE' => $transactionTmp->transactiontmp_BOE,
                'transactiontmpdt_billOfEntry' => $transactionTmp->transactiontmp_billOfEntry,
                'item_unit' => $itemUnit,
                'item_pack' => $itemPack,
                'item_qtyPerPack' => $qtyPerPack
            ];
    
            $validRecords[] = $transactionDtArray;
        }
    
        if (empty($invalidData)) {
            transactiondttmp::insert($validRecords);
        }
    
        $result = [
            'data' => 'success',
            'invalidData' => $invalidData,
        ];
    
        return ApiController::successResponse($result, 200);
    }
    
}
